/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
import { buildClientSchema, findBreakingChanges, findDangerousChanges, getIntrospectionQuery } from 'graphql';
import fetch from 'node-fetch';

const levelCritical = 1,
    levelWarning = 3;

/**
 * Format output
 *
 * @param {Number} severity
 * @param {*[]} changes
 * @returns {*[]}
 */
function buildOutput(severity, changes) {
    let output = [];

    for (let elem in changes) {
        if (changes.hasOwnProperty(elem)) {
            output.push({
                'severity': severity,
                'type': changes[elem].type,
                'description': changes[elem].description
            });
        }
    }

    return output;
}

/**
 * Load GraphQL schema from the endpoint url
 *
 * @param {String} url
 * @returns {Promise<GraphQLSchema>}
 */
async function loadGraphQlSchema(url) {
    let response = await fetch(url, {
        method: 'POST',
        headers: {
            'content-type': 'application/json'
        },
        body: JSON.stringify({
            query: getIntrospectionQuery()
        })
    });

    if (response.status === 200) {
        let {data} = await response.json();

        return buildClientSchema(data);
    }

    throw new Error(response.status);
}

/**
 * Retrieve GraphQL schema backward incompatible changes
 *
 * @param {Object} oldSchema
 * @param {Object} newSchema
 * @returns {*[]}
 */
function getCritical(oldSchema, newSchema) {
    try {
        let critical = findBreakingChanges(oldSchema, newSchema);

        return buildOutput(levelCritical, critical);
    } catch (e) {
        process.stderr.write(`GraphQL analysis failed: ${e.message}`);
    }
}

/**
 * Retrieve GraphQL schema difference warnings
 *
 * @param {Object} oldSchema
 * @param {Object} newSchema
 * @returns {*[]}
 */
function getWarnings(oldSchema, newSchema) {
    try {
        let warnings = findDangerousChanges(oldSchema, newSchema);

        return buildOutput(levelWarning, warnings);
    } catch (e) {
        process.stderr.write(`GraphQL analysis failed: ${e.message}`);
    }
}

/**
 * Execute the GraphQL schema comparison
 */
async function execute() {
    try {
        const args = process.argv.slice(2),
            oldSchemaUrl = args[0],
            newSchemaUrl = args[1],
            oldSchema = await loadGraphQlSchema(oldSchemaUrl),
            newSchema = await loadGraphQlSchema(newSchemaUrl);

        let critical = getCritical(oldSchema, newSchema),
            warnings = getWarnings(oldSchema, newSchema),
            result = critical.concat(warnings);

        process.stdout.write(JSON.stringify(result));
        process.exit(0);
    } catch (e) {
        process.stderr.write(`GraphQL analysis failed: ${e.message}`);
        process.exit(1);
    }
}

export default {
    execute,
    getCritical,
    getWarnings
};
