<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility;

use Sut\Domain\Compatibility\Analyzers\AnalyzerInterface;
use Sut\Domain\Issue\DTO\Issue;
use Sut\Domain\MRay\DTO\Module;

/**
 * Iterates over module core dependency usages and applies Analyzers to identify compatiblity issues
 */
class AnalyzeModule
{
    /**
     * @var AnalyzerInterface[]
     */
    private $analyzers;

    /**
     * @param AnalyzerInterface[] $analyzers
     */
    public function __construct(array $analyzers)
    {
        $this->analyzers = $analyzers;
    }

    /**
     * @param Module $module
     * @param string $version
     * @return Issue[]
     */
    public function execute(Module $module, string $version): array
    {
        $issues = [];
        foreach ($module->getDependencies() as $dependency) {
            foreach ($this->getFilteredUsages($dependency->getUsages()) as $usage) {
                foreach ($this->analyzers as $analyzer) {
                    $usageIssues = $analyzer->analyze($dependency, $usage, $version);
                    foreach ($usageIssues as $issue) {
                        $issues[$issue->getUniqueId()] = $issue;
                    }
                }
            }
        }

        return $issues;
    }

    /**
     * @param Module\DependencyUsage[] $usages
     * @return Module\DependencyUsage[]
     */
    private function getFilteredUsages(array $usages): array
    {
        $filteredUsages = [];
        foreach ($usages as $usage) {
            $index = $usage->getFile() . $usage->getPosition();
            if (!isset($filteredUsages[$index]) || $usage->getType() !== 'reference') {
                $filteredUsages[$index] = $usage;
            }
        }
        return $filteredUsages;
    }
}
