<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
namespace Sut\Domain\Compatibility\Analyzers\Db;

use Magento\Mray\MagentoApiIndex\Api\DbSchemaInfo;
use Sut\Domain\Compatibility\Analyzers\AnalyzerInterface;
use Sut\Domain\Issue\IssueFactory;
use Sut\Domain\MRay\DTO\Module;

class ColumnAnalyzer implements AnalyzerInterface
{
    private const USAGES = [
        'added-column-to-core-table',
        'extended-core-table-column',
    ];

    private const ISSUE_CODES = [
        'Added table column conflicts with core table column name' => 7003,
        'Extended core table column does not exist' => 7004
    ];

    /**
     * @var DbSchemaInfo
     */
    private $dbSchemaInfo;

    /**
     * @var IssueFactory
     */
    private $issueFactory;

    /**
     * @param IssueFactory $issueFactory
     * @param DbSchemaInfo $dbSchemaInfo
     */
    public function __construct(IssueFactory $issueFactory, DbSchemaInfo $dbSchemaInfo)
    {
        $this->issueFactory = $issueFactory;
        $this->dbSchemaInfo = $dbSchemaInfo;
    }

    /**
     * @inheritdoc
     */
    public function analyze(Module\Dependency $dependency, Module\DependencyUsage $usage, string $version): array
    {
        if ($dependency->getType() !== 'db') {
            return [];
        }

        if (!in_array($usage->getType(), self::USAGES)) {
            return [];
        }

        $tableColumnName = explode('.', $dependency->getRealName());
        if (!count($tableColumnName)) {
            return [];
        }

        $tableName = reset($tableColumnName);
        $columnName = end($tableColumnName);

        if ($usage->getType() === 'extended-core-table-column'
            && $this->dbSchemaInfo->getColumn($tableName, $columnName, $version) === null
        ) {
            return [$this->issueFactory->createIssue(
                self::ISSUE_CODES['Extended core table column does not exist'],
                [$dependency->getRealName(), $version],
                $usage->getPosition(),
                $usage->getFile()
            )];
        }

        if ($usage->getType() === 'added-column-to-core-table'
            && $this->dbSchemaInfo->getColumn($tableName, $columnName, $version) !== null
        ) {
            return [$this->issueFactory->createIssue(
                self::ISSUE_CODES['Added table column conflicts with core table column name'],
                [$dependency->getRealName(), $version],
                $usage->getPosition(),
                $usage->getFile()
            )];
        }

        return [];
    }
}
