<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
namespace Sut\Domain\Compatibility\Analyzers\Php;

use Sut\Domain\Compatibility\Analyzers\AnalyzerInterface;
use Sut\Domain\Compatibility\GetCodeGenerationBase;
use Sut\Domain\Compatibility\Index;
use Sut\Domain\Issue\IssueFactory;
use Sut\Domain\MRay\DTO\Module;

final class ClassAnalyzer implements AnalyzerInterface
{
    private const USAGES = [
        'extend',
        'imported',
        'used',
        'loaded',
        'reference'
    ];

    private const NON_API_CODES =  [
        'extend' => 1121,
        'imported' => 1122,
        'used' => 1124,
        'loaded' => 1123,
        'reference' => 1124
    ];

    private const NON_EXISTENT_CODES = [
        'extend' => 1111,
        'imported' => 1112,
        'used' => 1114,
        'loaded' => 1113,
        'reference' => 1114
    ];

    private const DEPRECATED_CODES = [
        'extend' => 1131,
        'imported' => 1132,
        'used' => 1134,
        'loaded' => 1133,
        'reference' => 1134
    ];

    private const USE_INTERFACE_INSTEAD_CODE = 1104;

    /**
     * @var Index
     */
    private $index;

    /**
     * @var IssueFactory
     */
    private $issueFactory;

    /**
     * @var GetCodeGenerationBase
     */
    private $getCodeGenerationBase;

    /**
     * @param Index $index
     * @param IssueFactory $issueFactory
     * @param GetCodeGenerationBase $getCodeGenerationBase
     */
    public function __construct(
        Index                 $index,
        IssueFactory          $issueFactory,
        GetCodeGenerationBase $getCodeGenerationBase
    ) {
        $this->index = $index;
        $this->issueFactory = $issueFactory;
        $this->getCodeGenerationBase = $getCodeGenerationBase;
    }

    /**
     * @inheritdoc
     */
    public function analyze(Module\Dependency $dependency, Module\DependencyUsage $usage, string $version): array
    {
        $issues = [];

        if ($dependency->getType() !== 'class') {
            return $issues;
        }

        if (!in_array($usage->getType(), self::USAGES)) {
            return $issues;
        }

        $fqn = $dependency->getFqn();

        if (!$this->index->isPresent($fqn, $version)) {
            $originalExtensionClass = $dependency->getOriginalExtensionClass();
            if ($originalExtensionClass &&
                $originalExtensionClass !== $fqn &&
                $this->index->isApi($originalExtensionClass, $version)
            ) {
                $issues[] = $this->issueFactory->createIssue(
                    self::USE_INTERFACE_INSTEAD_CODE,
                    [$fqn, $fqn . 'Interface'],
                    $usage->getPosition(),
                    $usage->getFile()
                );
                return $issues;
            }

            $autogenerationBaseName = $this->getCodeGenerationBase->execute($fqn);

            if (!$autogenerationBaseName ||
                $autogenerationBaseName === $fqn ||
                !$this->index->isPresent($autogenerationBaseName, $version)
            ) {
                $issues[] = $this->issueFactory->createIssue(
                    self::NON_EXISTENT_CODES[$usage->getType()],
                    [$fqn, $version],
                    $usage->getPosition(),
                    $usage->getFile()
                );
                return $issues;
            }
            $fqn = $autogenerationBaseName;
        }

        if (!$this->index->isApi($fqn, $version)) {
            $issues[] = $this->issueFactory->createIssue(
                self::NON_API_CODES[$usage->getType()],
                [$fqn, $version],
                $usage->getPosition(),
                $usage->getFile()
            );
        }

        if ($this->index->isDeprecated($fqn, $version)) {
            $issues[] = $this->issueFactory->createIssue(
                self::DEPRECATED_CODES[$usage->getType()],
                [$fqn, $version, $this->index->getDeprecatedRecommendation($fqn, $version)],
                $usage->getPosition(),
                $usage->getFile()
            );
        }

        return $issues;
    }
}
