<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
namespace Sut\Domain\Compatibility\Analyzers\Php;

use Sut\Domain\Compatibility\Analyzers\AnalyzerInterface;
use Sut\Domain\Compatibility\Index;
use Sut\Domain\Issue\IssueFactory;
use Sut\Domain\MRay\DTO\Module;

final class ConstantAnalyzer implements AnalyzerInterface
{
    private const USAGES = [
        'fetch',
        'assign',
        'override',
    ];

    private const NON_API_CODES = [
        'fetch' => 1224,
        'assign' => 1226,
        'override' => 1225,
    ];

    private const NON_EXISTENT_CODES = [
        'fetch' => 1214,
        'assign' => 1216,
        'override' => 1215,
    ];

    private const DEPRECATED_CODES = [
        'fetch' => 1234,
        'assign' => 1236,
        'override' => 1235,
    ];

    /**
     * @var Index
     */
    private $index;

    /**
     * @var IssueFactory
     */
    private $issueFactory;

    /**
     * PropertyAnalyzer constructor.
     * @param Index $index
     * @param IssueFactory $issueFactory
     */
    public function __construct(Index $index, IssueFactory $issueFactory)
    {
        $this->index = $index;
        $this->issueFactory = $issueFactory;
    }

    /**
     * @inheritdoc
     */
    public function analyze(Module\Dependency $dependency, Module\DependencyUsage $usage, string $version): array
    {
        $issues = [];

        if ($dependency->getType() !== 'constant') {
            return $issues;
        }

        if (!in_array($usage->getType(), self::USAGES)) {
            return $issues;
        }

        $fqn = $dependency->getFqn();
        if (!$this->index->isPresent($fqn, $version)) {
            $issues[] = $this->issueFactory->createIssue(
                self::NON_EXISTENT_CODES[$usage->getType()],
                [$fqn, $version],
                $usage->getPosition(),
                $usage->getFile()
            );
            return $issues;
        }

        if (!$this->index->isApi($fqn, $version)) {
            $issues[] = $this->issueFactory->createIssue(
                self::NON_API_CODES[$usage->getType()],
                [$fqn, $version],
                $usage->getPosition(),
                $usage->getFile()
            );
        }

        if ($this->index->isDeprecated($fqn, $version)) {
            $issues[] = $this->issueFactory->createIssue(
                self::DEPRECATED_CODES[$usage->getType()],
                [$fqn, $version, $this->index->getDeprecatedRecommendation($fqn, $version)],
                $usage->getPosition(),
                $usage->getFile()
            );
        }
        return $issues;
    }
}
