<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
namespace Sut\Domain\Compatibility\Analyzers\Php;

use Sut\Domain\Compatibility\Analyzers\AnalyzerInterface;
use Sut\Domain\Compatibility\GetCodeGenerationBase;
use Sut\Domain\Compatibility\Index;
use Sut\Domain\Issue\IssueFactory;
use Sut\Domain\MRay\DTO\Module;

final class InterfaceAnalyzer implements AnalyzerInterface
{
    private const USAGES = [
        'inherited',
        'implement',
        'imported',
        'used',
        'reference'
    ];

    private const NON_API_CODES =  [
        'inherited' => 1327,
        'implement' => 1328,
        'imported' => 1322,
        'used' => 1324,
        'reference' => 1324
    ];

    private const NON_EXISTENT_CODES =  [
        'inherited' => 1317,
        'implement' => 1318,
        'imported' => 1312,
        'used' => 1314,
        'reference' => 1314
    ];

    private const DEPRECATED_CODES = [
        'inherited' => 1337,
        'implement' => 1338,
        'imported' => 1332,
        'used' => 1334,
        'reference' => 1334,
    ];

    /**
     * @var Index
     */
    private $index;

    /**
     * @var IssueFactory
     */
    private $issueFactory;

    /**
     * @var GetCodeGenerationBase
     */
    private $getCodeGenerationBase;

    /**
     * @param Index $index
     * @param IssueFactory $issueFactory
     * @param GetCodeGenerationBase $getCodeGenerationBase
     */
    public function __construct(
        Index                 $index,
        IssueFactory          $issueFactory,
        GetCodeGenerationBase $getCodeGenerationBase
    ) {
        $this->index = $index;
        $this->issueFactory = $issueFactory;
        $this->getCodeGenerationBase = $getCodeGenerationBase;
    }

    /**
     * @inheritdoc
     */
    public function analyze(Module\Dependency $dependency, Module\DependencyUsage $usage, string $version): array
    {
        $issues = [];

        if ($dependency->getType() !== 'interface') {
            return $issues;
        }

        if (!in_array($usage->getType(), self::USAGES)) {
            return $issues;
        }

        $fqn = $dependency->getFqn();
        if (!$this->index->isPresent($fqn, $version)) {
            $originalExtensionInterface = $dependency->getOriginalExtensionInterface();
            if ($originalExtensionInterface &&
                $originalExtensionInterface !== $fqn &&
                $this->index->isPresent($originalExtensionInterface, $version)
            ) {
                $fqn = $originalExtensionInterface;
            }
            $baseInterface = $this->getCodeGenerationBase->execute($fqn);
            if ($baseInterface &&
                $baseInterface !== $fqn &&
                $this->index->isPresent($baseInterface, $version)
            ) {
                $fqn = $baseInterface;
            }
            if ($fqn === $dependency->getFqn()) {
                $issues[] = $this->issueFactory->createIssue(
                    self::NON_EXISTENT_CODES[$usage->getType()],
                    [$fqn, $version],
                    $usage->getPosition(),
                    $usage->getFile()
                );
                return $issues;
            }
        }

        if (!$this->index->isApi($fqn, $version)) {
            $issues[] = $this->issueFactory->createIssue(
                self::NON_API_CODES[$usage->getType()],
                [$fqn, $version],
                $usage->getPosition(),
                $usage->getFile()
            );
        }

        if ($this->index->isDeprecated($fqn, $version)) {
            $issues[] = $this->issueFactory->createIssue(
                self::DEPRECATED_CODES[$usage->getType()],
                [$fqn, $version, $this->index->getDeprecatedRecommendation($fqn, $version)],
                $usage->getPosition(),
                $usage->getFile()
            );
        }

        return $issues;
    }
}
