<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility\Analyzers\Php\MethodAnalyzer;

class MatchTypes
{
    /**
     * @param string $firstType
     * @param string $secondType
     * @return bool
     */
    public function execute(string $firstType, string $secondType): bool
    {
        if (!$this->isTypeIdentified($firstType) || !$this->isTypeIdentified($secondType)) {
            return true;
        }
        if ($this->isClass($firstType) && $this->isClass($secondType)) {
            return true;
        }
        if ($this->isArray($firstType) && $this->isArray($secondType)) {
            return true;
        }
        if ($this->haveCommonTypes($firstType, $secondType)) {
            return true;
        }
        return false;
    }

    /**
     * @param string $type
     * @return bool
     */
    private function isArray(string $type): bool
    {
        return strpos($type, 'array') !== false || strpos($type, '[]') !== false;
    }

    /**
     * @param string $type
     * @return bool
     */
    private function isClass(string $type): bool
    {
        return strpos($type, '\\') !== false;
    }

    /**
     * @param string $type
     * @return bool
     */
    private function isTypeIdentified(string $type): bool
    {
        return $type
            && strpos($type, 'mixed') === false
            && strpos($type, 'null') === false
            && strpos($type, 'callable') === false
            && strpos($type, '$this') === false;
    }

    /**
     * @param string $type
     * @return array
     */
    private function getTypes(string $type): array
    {
        $types = explode('|', trim($type, '()\\ '));
        return array_map(
            function ($type) {
                return trim($type, '()\\ ');
            },
            $types
        );
    }

    /**
     * @param string $firstType
     * @param string $secondType
     * @return bool
     */
    private function haveCommonTypes(string $firstType, string $secondType): bool
    {
        return !empty(array_intersect($this->getTypes($firstType), $this->getTypes($secondType)));
    }
}
