<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility\Data;

use DateTime;
use Exception;
use Sut\Domain\Time\TimerInterface;

class AnalysisRequest
{
    /**
     * @var string
     */
    private $analysisPath;

    /**
     * @var string
     */
    private $currentVersion;

    /**
     * @var string
     */
    private $comingVersion;

    /**
     * @var string|null
     */
    private $jsonOutputPath;

    /**
     * @var string|null
     */
    private $htmlOutputPath;

    /**
     * @var string|null
     */
    private $context;

    /**
     * @var int
     */
    private $minIssueLevel;

    /**
     * @var DateTime
     */
    private $creationTime;

    /**
     * @param string $analysisPath
     * @param string $currentVersion
     * @param string $comingVersion
     * @param string|null $context
     * @param int $minIssueLevel
     * @param string|null $jsonOutputPath
     * @param string|null $htmlOutputPath
     * @param TimerInterface $timer
     * @throws Exception
     */
    public function __construct(
        string $analysisPath,
        string $currentVersion,
        string $comingVersion,
        ?string $context,
        int $minIssueLevel,
        ?string $jsonOutputPath,
        ?string $htmlOutputPath,
        TimerInterface $timer
    ) {
        $this->analysisPath = $analysisPath;
        $this->currentVersion = $currentVersion;
        $this->comingVersion = $comingVersion;
        $this->context = $context;
        $this->jsonOutputPath = $jsonOutputPath;
        $this->htmlOutputPath = $htmlOutputPath;
        $this->minIssueLevel = $minIssueLevel;
        $this->creationTime = new DateTime('@' . $timer->time());
    }

    /**
     * @return string
     */
    public function getAnalysisPath(): string
    {
        return $this->analysisPath;
    }

    /**
     * @return string
     */
    public function getCurrentVersion(): string
    {
        return $this->currentVersion;
    }

    /**
     * @return string
     */
    public function getComingVersion(): string
    {
        return $this->comingVersion;
    }

    /**
     * @return string|null
     */
    public function getContext(): ?string
    {
        return $this->context;
    }

    /**
     * @return DateTime
     */
    public function getCreationTime(): DateTime
    {
        return $this->creationTime;
    }

    /**
     * @return string|null
     */
    public function getJsonOutputPath(): ?string
    {
        return $this->jsonOutputPath;
    }

    /**
     * @return string|null
     */
    public function getHtmlOutputPath(): ?string
    {
        return $this->htmlOutputPath;
    }

    /**
     * @return int
     */
    public function getMinIssueLevel(): int
    {
        return $this->minIssueLevel;
    }
}
