<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility\Data;

use DateTime;
use Exception;
use Sut\Domain\Compatibility\GetMemoryPeakUsage;
use Sut\Domain\Issue\DTO\Issue;
use Sut\Domain\Time\TimerInterface;

class AnalysisResult
{
    /**
     * @var ModuleResult[]
     */
    private $modules;

    /**
     * @var Issue[]
     */
    private $issuesCache;

    /**
     * @var FileResult[]
     */
    private $filesCache;

    /**
     * @var AnalysisRequest
     */
    private $analysisRequest;

    /**
     * @var DateTime
     */
    private $creationTime;

    /**
     * @var IssuesSummaryResult
     */
    private $summary;

    /**
     * @var GetMemoryPeakUsage
     */
    private $getMemoryPeakUsage;

    /**
     * @param array $modules
     * @param AnalysisRequest $analysisRequest
     * @param TimerInterface $timer
     * @param GetMemoryPeakUsage $getMemoryPeakUsage
     * @throws Exception
     */
    public function __construct(
        array $modules,
        AnalysisRequest $analysisRequest,
        TimerInterface $timer,
        GetMemoryPeakUsage $getMemoryPeakUsage
    ) {
        $this->modules = $modules;
        $this->analysisRequest = $analysisRequest;
        $this->creationTime = new DateTime('@' . $timer->time());
        $this->getMemoryPeakUsage = $getMemoryPeakUsage;
    }

    /**
     * @return ModuleResult[]
     */
    public function getModules(): array
    {
        return $this->modules;
    }

    /**
     * @return Issue[]
     */
    public function getIssues(): array
    {
        if (!isset($this->issuesCache)) {
            $issues = [];
            foreach ($this->modules as $module) {
                $issues[] = $module->getIssues();
            }
            $issues = array_merge([], ...$issues);

            $this->issuesCache = [];
            /** @var Issue $issue */
            foreach ($issues as $issue) {
                $this->issuesCache[$issue->getUniqueId()] = $issue;
            }

        }
        return $this->issuesCache;
    }

    /**
     * @return FileResult[]
     */
    public function getFiles(): array
    {
        if (!isset($this->filesCache)) {
            $files = [];
            foreach ($this->modules as $module) {
                $files[] = $module->getFiles();
            }
            $this->filesCache = array_merge([], ...$files);
        }
        return $this->filesCache;
    }

    /**
     * @return IssuesSummaryResult
     */
    public function getSummary(): IssuesSummaryResult
    {
        if (!$this->summary) {
            $this->summary = new IssuesSummaryResult(
                $this->getIssues(),
                $this->getAnalysisRequest()->getMinIssueLevel()
            );
        }
        return $this->summary;
    }

    /**
     * @return AnalysisRequest
     */
    public function getAnalysisRequest(): AnalysisRequest
    {
        return $this->analysisRequest;
    }

    /**
     * @return DateTime
     */
    public function getCreationTime(): DateTime
    {
        return $this->creationTime;
    }

    /**
     * @return float
     */
    public function getMemoryPeakUsage(): float
    {
        return $this->getMemoryPeakUsage->execute();
    }

    /**
     * @return int
     */
    public function getTotalNumberOfFiles(): int
    {
        $totalFiles = 0;
        foreach ($this->getModules() as $module) {
            $totalFiles += $module->getTotalNumberOfFiles();
        }
        return $totalFiles;
    }

    /**
     * @return string[]
     */
    public function getIncompatibleFileLevels(): array
    {
        $files = [];
        foreach ($this->getModules() as $module) {
            $files[] = $module->getIncompatibleFileLevels();
        }
        return array_merge([], ...$files);
    }

    /**
     * @return string
     */
    public function getFilesSummary(): string
    {
        if ($this->getTotalNumberOfFiles() === 0) {
            return '00.00%';
        }
        $numberOfIncompatibleFiles = count($this->getIncompatibleFileLevels());
        $percent = ($numberOfIncompatibleFiles / $this->getTotalNumberOfFiles()) * 100;
        return  sprintf('%0.2f%% (%s/%s)', $percent, $numberOfIncompatibleFiles, $this->getTotalNumberOfFiles());
    }

    /**
     * @return string
     */
    public function getModulesSummary(): string
    {
        $totalModulesNumber = count($this->getModules());
        if ($totalModulesNumber === 0) {
            return '00.00%';
        }
        $percent = ($this->getIncompatibleModulesNumber() / $totalModulesNumber) * 100;
        return  sprintf('%0.2f%% (%s/%s)', $percent, $this->getIncompatibleModulesNumber(), $totalModulesNumber);
    }

    /**
     * @return int
     */
    public function getIncompatibleModulesNumber(): int
    {
        $incompatibleModulesNumber = 0;
        foreach ($this->getModules() as $module) {
            if (!empty($module->getIssues())) {
                $incompatibleModulesNumber++;
            }
        }
        return $incompatibleModulesNumber;
    }
}
