<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility\Data;

use Sut\Domain\Issue\DTO\Issue;

class FileResult
{
    /**
     * @var string
     */
    private $path;

    /**
     * @var Issue[]
     */
    private $issues;

    /**
     * @var int
     */
    private $minIssueLevel;

    /**
     * @var Issue[]
     */
    private $filteredIssues;

    /**
     * @var IssuesSummaryResult
     */
    private $summary;

    /**
     * @param string $path
     * @param array $issues
     * @param int $minIssueLevel
     */
    public function __construct(string $path, array $issues, int $minIssueLevel)
    {
        $this->path = $path;
        $this->issues = $issues;
        $this->minIssueLevel = $minIssueLevel;
    }

    /**
     * @return Issue[]
     */
    public function getIssues(): array
    {
        return $this->issues;
    }

    /**
     * @return Issue[]
     */
    public function getFilteredIssues(): array
    {
        if (!$this->filteredIssues) {
            $this->filteredIssues = [];
            foreach ($this->getIssues() as $issue) {
                if ($issue->getLevel() <= $this->minIssueLevel) {
                    $this->filteredIssues[] = $issue;
                }
            }
        }

        return $this->filteredIssues;
    }

    /**
     * @return string
     */
    public function getPath(): string
    {
        return $this->path;
    }

    /**
     * @return IssuesSummaryResult
     */
    public function getSummary(): IssuesSummaryResult
    {
        if (!$this->summary) {
            $this->summary = new IssuesSummaryResult(
                $this->getIssues(),
                $this->minIssueLevel
            );
        }
        return $this->summary;
    }
}
