<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility\Data;

use Sut\Domain\Issue\DTO\Issue;
use Sut\Domain\Issue\DTO\IssueLevel;

class IssuesSummaryResult
{
    /**
     * @var Issue[]
     */
    private $issues;

    /**
     * @var int
     */
    private $minIssueLevel;

    /**
     * @var Issue[]
     */
    private $filteredIssues;

    /**
     * @var Issue[]
     */
    private $criticals = [];

    /**
     * @var Issue[]
     */
    private $errors = [];

    /**
     * @var Issue[]
     */
    private $warnings = [];

    /**
     * @param Issue[] $issues
     * @param int $minIssueLevel
     */
    public function __construct(array $issues, int $minIssueLevel)
    {
        $this->issues = $issues;
        $this->minIssueLevel = $minIssueLevel;
        $this->prepareSummary($issues);
    }

    /**
     * @return Issue[]
     */
    public function getIssues(): array
    {
        usort($this->issues, function (Issue $issue1, Issue $issue2) {
            if ($issue1->getFileName() == $issue2->getFileName()) {
                return $issue1->getLineNumber() <=> $issue2->getLineNumber();
            }
            return $issue1->getFileName() <=> $issue2->getFileName();
        });

        return $this->issues;
    }

    /**
     * @return Issue[]
     */
    public function getFilteredIssues(): array
    {
        if (!$this->filteredIssues) {
            $this->filteredIssues = [];
            foreach ($this->getIssues() as $issue) {
                if ($issue->getLevel() <= $this->minIssueLevel) {
                    $this->filteredIssues[] = $issue;
                }
            }
        }

        return $this->filteredIssues;
    }

    /**
     * @return Issue[]
     */
    public function getCriticals(): array
    {
        return $this->criticals;
    }

    /**
     * @return Issue[]
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * @return array
     */
    public function getWarnings(): array
    {
        return $this->warnings;
    }

    /**
     * @return int
     */
    public function getMinIssueLevel(): int
    {
        return $this->minIssueLevel;
    }

    /**
     * @param Issue[] $issues
     */
    private function prepareSummary(array $issues): void
    {
        foreach ($issues as $issue) {
            if ($issue->getLevel() === IssueLevel::WARNING) {
                $this->warnings[] = $issue;
            }
            if ($issue->getLevel() === IssueLevel::ERROR) {
                $this->errors[] = $issue;
            }
            if ($issue->getLevel() === IssueLevel::CRITICAL) {
                $this->criticals[] = $issue;
            }
        }
    }
}
