<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility;

use Exception;
use InvalidArgumentException;
use Magento\Mray\Api\GetProjectVersion;
use Sut\Domain\Compatibility\Data\AnalysisRequest;
use Sut\Domain\Compatibility\Data\AnalysisResult;
use Sut\Domain\Compatibility\Data\AnalysisResultFactory;
use Sut\Domain\Compatibility\Data\FileResult;
use Sut\Domain\Compatibility\Data\ModuleResult;
use Sut\Domain\Eslint\GetEslintIssues;
use Sut\Domain\Issue\DTO\Issue;
use Sut\Domain\PhpCs\GetPhpCsIssues;
use Symfony\Component\Console\Style\SymfonyStyle;

class GetIssues
{
    /**
     * @var AnalyzeModule
     */
    private $analyzeModule;

    /**
     * @var Index
     */
    private $mRay;

    /**
     * @var GetPhpCsIssues
     */
    private $getPhpCsIssues;

    /**
     * @var GetEslintIssues
     */
    private $getEslintIssues;

    /**
     * @var AnalysisResultFactory
     */
    private $analysisResultFactory;

    /**
     * @param AnalyzeModule $iteratorService
     * @param Index $index
     * @param GetPhpCsIssues $getPhpCsIssues
     * @param GetEslintIssues $getEslintIssues
     * @param AnalysisResultFactory $analysisResultFactory
     */
    public function __construct(
        AnalyzeModule         $iteratorService,
        Index                 $index,
        GetPhpCsIssues        $getPhpCsIssues,
        GetEslintIssues       $getEslintIssues,
        AnalysisResultFactory $analysisResultFactory
    ) {
        $this->analyzeModule = $iteratorService;
        $this->mRay = $index;
        $this->getPhpCsIssues = $getPhpCsIssues;
        $this->getEslintIssues = $getEslintIssues;
        $this->analysisResultFactory = $analysisResultFactory;
    }

    /**
     * @param SymfonyStyle $output
     * @param AnalysisRequest $analysisRequest
     * @param bool $ignoreCurrentVersionIssues
     * @return AnalysisResult
     */
    public function execute(
        SymfonyStyle $output,
        AnalysisRequest $analysisRequest,
        bool $ignoreCurrentVersionIssues = false
    ): AnalysisResult {
        if ($ignoreCurrentVersionIssues) {
            $output->writeln('It might take a while');
            $result = $this->checkVersion(
                $output,
                $analysisRequest,
                [],
                true
            );

            return $this->checkVersion(
                $output,
                $analysisRequest,
                $result->getIssues()
            );
        }
        return $this->checkVersion(
            $output,
            $analysisRequest,
            []
        );
    }

    /**
     * @param SymfonyStyle $output
     * @param AnalysisRequest $analysisRequest
     * @param Issue[] $ignoredIssues
     * @param bool $currentVersionRun
     * @return AnalysisResult
     * @throws Exception
     */
    private function checkVersion(
        SymfonyStyle $output,
        AnalysisRequest $analysisRequest,
        array $ignoredIssues = [],
        bool $currentVersionRun = false
    ): AnalysisResult {
        $comingVersion = $analysisRequest->getComingVersion();

        $this->validateVersion($comingVersion);

        if ($currentVersionRun) {
            $currentVersion = $analysisRequest->getCurrentVersion();
            $this->validateVersion($currentVersion);
            $comingVersion = $currentVersion;
        }

        $path = $analysisRequest->getAnalysisPath();
        $version = $analysisRequest->getCurrentVersion() ?: (new GetProjectVersion())->execute($path);

        $installedModules = $this->mRay->getModules($path, $version);

        $numberOfModules = count($installedModules);

        if ($numberOfModules === 0) {
            $output->getErrorStyle()->writeln(
                'No custom modules found in ' . $analysisRequest->getAnalysisPath()
            );
            return $this->analysisResultFactory->create([], $analysisRequest);
        }

        $output->writeln(sprintf('Analysing %d module(s)...', $numberOfModules));
        $modules = [];
        foreach ($installedModules as $installedModule) {
            $foundIssues = [];
            $foundIssuesCounter = 0;

            // phpcs:ignore
            $issues = array_merge(
                $this->analyzeModule->execute($installedModule, $comingVersion),
                $this->getPhpCsIssues->execute($installedModule->getPath()),
                $this->getEslintIssues->execute($installedModule->getPath())
            );

            foreach ($issues as $issue) {
                if (!isset($ignoredIssues[$issue->getUniqueId()])) {
                    $foundIssues[$issue->getUniqueId()] = $issue;
                }
                $foundIssuesCounter++;
            }

            $output->write(' * ' . $installedModule->getName());
            $output->writeln(sprintf(' - found %d issue(s)', $foundIssuesCounter));

            $modules[] = new ModuleResult(
                $installedModule,
                $this->getFiles($foundIssues, $analysisRequest->getMinIssueLevel()),
                $analysisRequest->getMinIssueLevel()
            );
        }
        return $this->analysisResultFactory->create($modules, $analysisRequest);
    }

    /**
     * @param Issue[] $foundIssues
     * @param int $minIssueLevel
     * @return FileResult[]
     */
    private function getFiles(array $foundIssues, int $minIssueLevel): array
    {
        $filesIssues = [];
        foreach ($foundIssues as $issue) {
            $filesIssues[$issue->getFileName()][] = $issue;
        }

        $files = [];
        foreach ($filesIssues as $path => $fileIssues) {
            $files[] = new FileResult($path, $fileIssues, $minIssueLevel);
        }

        return $files;
    }

    /**
     * @param string $version
     */
    private function validateVersion(string $version): void
    {
        if (!in_array($version, $this->mRay->getAvailableVersions())) {
            throw new InvalidArgumentException(
                sprintf(
                    'Upgrade check for Adobe Commerce version %s is not available. ' .
                    'Available Adobe Commerce versions: %s',
                    $version,
                    implode(' | ', $this->mRay->getAvailableVersions())
                )
            );
        }
    }
}
