<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility;

use Magento\Mray\Api\CustomProjectIndex;
use Magento\Mray\MagentoApiIndex\Api;
use Magento\Mray\MagentoApiIndex\Api\GetIndexedVersions;
use Magento\Mray\MagentoApiIndex\Index\Version;
use Sut\Domain\MRay\DTO\MethodSignature;
use Sut\Domain\MRay\DTO\Module;

class Index
{
    /**
     * @var array
     */
    private $apiIndexCache = [];

    /**
     * @var Module[]
     */
    private $moduleCache = [];

    /**
     * @var Api
     */
    private $apiIndex;

    /**
     * @var CustomProjectIndex|null
     */
    private $customIndex;

    /**
     * @param Api $indexApi
     * @param CustomProjectIndex|null $customIndex
     */
    public function __construct(Api $indexApi, CustomProjectIndex $customIndex = null)
    {
        $this->apiIndex = $indexApi;
        $this->customIndex = $customIndex ?: new CustomProjectIndex();
    }

    /**
     * @return array
     */
    public function getAvailableVersions(): array
    {
        return (new GetIndexedVersions())->execute();
    }

    /**
     * @param string $reference
     * @param string $version
     * @return bool
     */
    public function isPresent(string $reference, string $version): bool
    {
        return $this->getApiIndex($version)->isApi($reference) !== null;
    }

    /**
     * @param string $reference
     * @param string $version
     * @return bool
     */
    public function isApi(string $reference, string $version): bool
    {
        return (bool) $this->getApiIndex($version)->isApi($reference);
    }

    /**
     * @param string $reference
     * @param string $version
     * @return bool
     */
    public function isDeprecated(string $reference, string $version): bool
    {
        return $this->getApiIndex($version)->isDeprecated($reference);
    }

    /**
     * @param string $reference
     * @param string $version
     * @return string
     */
    public function getDeprecatedRecommendation(string $reference, string $version): string
    {
        $recommendation = $this->getApiIndex($version)->getDeprecationDeclaration($reference)['see'] ?? '';
        return $recommendation ? '- See ' . $recommendation : '';
    }

    /**
     * @param string $path
     * @param string $version
     * @return Module[]
     */
    public function getModules(string $path, string $version): array
    {
        if (empty($this->moduleCache)) {
            $this->moduleCache = $this->obtainModuleEntries($path, $version);
        }
        return $this->moduleCache;
    }

    /**
     * @param string $reference
     * @param string $version
     * @return MethodSignature|null
     */
    public function getMethodSignature(string $reference, string $version): ?MethodSignature
    {
        $details = $this->getApiIndex($version)->describe($reference);
        $paramsResult = [];

        if ($details === null || !isset($details['params'])) {
            return null;
        }

        foreach ($details['params'] as $data) {
            $paramsResult[] = new Module\Parameter(
                $data['type']['expr'] ?? '',
                0,
                $data['name'] ?? '',
                $data['optional']
            );
        }

        return new MethodSignature($paramsResult, $details['returns']['expr'] ?? '');
    }

    /**
     * @param string $path
     * @param string $version
     * @return Module[]
     */
    private function obtainModuleEntries(string $path, string $version): array
    {
        $customizations = $this->customIndex->execute($path, $version)->getCustomizations();

        $modulesResult = [];
        foreach ($customizations['modules'] as $data) {
            $modulesResult[] = new Module(
                $data['name'],
                $data['package'] ?? "",
                $data['version'] ?? "",
                $this->obtainDependencies($data['dependencies'] ?? []),
                $data['path']
            );
        }

        return $modulesResult;
    }

    /**
     * @param array $dependencies
     * @return array
     */
    private function obtainDependencies(array $dependencies): array
    {
        $processedDependencies = [];

        foreach ($dependencies as $dependencyFqn => $dependency) {
            $processedDependencies[] = $this->processDependency($dependencyFqn, $dependency);
        }

        return $processedDependencies;
    }

    /**
     * @param string $dependencyFqn
     * @param array $dependency
     * @return Module\Dependency
     */
    private function processDependency(string $dependencyFqn, array $dependency): Module\Dependency
    {
        return new Module\Dependency(
            $dependencyFqn,
            $dependency['type'],
            $dependency['realName'] ?? "",
            $dependency['module_name'] ?? "",
            $dependency['package'] ?? "",
            $this->processUsages($dependency['usage'] ?? [])
        );
    }

    /**
     * @param array $usages
     * @return array
     */
    private function processUsages(array $usages): array
    {
        $processedUsages = [];

        foreach ($usages as $usageType => $usage) {
            foreach ($usage as $usageEntry) {
                $processedUsages[] = $this->processUsage($usageType, $usageEntry);
            }
        }

        return $processedUsages;
    }

    /**
     * @param string $usageType
     * @param array $usage
     * @return Module\DependencyUsage
     */
    private function processUsage(string $usageType, array $usage): Module\DependencyUsage
    {
        return new Module\DependencyUsage($usageType, $usage);
    }

    /**
     * @param string $version
     * @return Version
     */
    private function getApiIndex(string $version): Version
    {
        if (!isset($this->apiIndexCache[$version])) {
            $this->apiIndexCache[$version] = $this->apiIndex->getApiIndex($version);
        }
        return $this->apiIndexCache[$version];
    }
}
