<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility\Output;

use Sut\Domain\Compatibility\Data\AnalysisResult;
use Sut\Domain\Compatibility\OutputInterface;
use Sut\Domain\Issue\DTO\Issue;
use Symfony\Component\Console\Formatter\OutputFormatterStyle;
use Symfony\Component\Console\Style\SymfonyStyle;

class ConsoleIssues implements OutputInterface
{
    /**
     * @param SymfonyStyle $output
     * @param AnalysisResult $analysisResult
     */
    public function write(
        SymfonyStyle $output,
        AnalysisResult $analysisResult
    ): void {
        $this->addStyles($output);

        if (empty($analysisResult->getSummary()->getFilteredIssues())) {
            $output->writeln('No problems found');
            return;
        }

        foreach ($analysisResult->getModules() as $moduleResult) {
            if (empty($moduleResult->getSummary()->getFilteredIssues())) {
                continue;
            }
            $output->section('Module Name: ' . $moduleResult->getModule()->getName());
            foreach ($moduleResult->getFiles() as $fileResult) {
                if (empty($fileResult->getSummary()->getFilteredIssues())) {
                    continue;
                }
                $issuesList = [];
                foreach ($fileResult->getSummary()->getFilteredIssues() as $issue) {
                    $issuesList[] = $this->getIssueLine($issue);
                }
                $output->section('File: ' . $fileResult->getPath());
                $output->listing($issuesList);
            }
        }

        if (!empty($analysisResult->getSummary()->getFilteredIssues())) {
            $output->writeln(sprintf(
                '<comment>Check %s for a detailed list of Upgrade Compatibility Tool errors.</comment>',
                'https://experienceleague.adobe.com/docs/commerce-operations/upgrade-guide/' .
                'upgrade-compatibility-tool/error-messages.html'
            ));
        }
    }

    /**
     * @param Issue $issue
     * @return string
     */
    private function getIssueLine(Issue $issue): string
    {
        $level = $issue->getLevelString();
        return implode(
            '',
            [
                '<' . strtolower($level) . '>',
                '[' . $level . ']',
                '</' . strtolower($level) . '>',
                '[' . $issue->getCode() . ']',
                ' Line ' . $issue->getLineNumber() . ':',
                ' ' . $issue->getMessage()
            ]
        );
    }

    /**
     * @param SymfonyStyle $output
     */
    public function addStyles(SymfonyStyle $output): void
    {
        $output->getFormatter()->setStyle('critical', new OutputFormatterStyle('red', null));
        $output->getFormatter()->setStyle('warning', new OutputFormatterStyle('yellow', null));
        $output->getFormatter()->setStyle('file', new OutputFormatterStyle('black', null));
    }
}
