<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility\Output;

use Sut\Domain\Compatibility\Data\AnalysisResult;
use Sut\Domain\Compatibility\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class ConsoleSummary implements OutputInterface
{
    /**
     * @param SymfonyStyle $output
     * @param AnalysisResult $analysisResult
     */
    public function write(
        SymfonyStyle $output,
        AnalysisResult $analysisResult
    ): void {
        $output->definitionList(
            ['Current version' => $analysisResult->getAnalysisRequest()->getCurrentVersion()],
            ['Target version' => $analysisResult->getAnalysisRequest()->getComingVersion()],
            ['Execution time' => $this->getRunningTime($analysisResult)],
            ['Modules that require update' => $analysisResult->getModulesSummary()],
            ['Files that require update' => $analysisResult->getFilesSummary()],
            ['Total critical issues' => count($analysisResult->getSummary()->getCriticals())],
            ['Total errors' => count($analysisResult->getSummary()->getErrors())],
            ['Total warnings' => count($analysisResult->getSummary()->getWarnings())],
            ['Memory peak usage' => $this->formatMemoryPeakUsage($analysisResult->getMemoryPeakUsage())]
        );
    }

    /**
     * @param AnalysisResult $analysisResult
     * @return string
     */
    private function getRunningTime(AnalysisResult $analysisResult): string
    {
        return $analysisResult->getCreationTime()
            ->diff($analysisResult->getAnalysisRequest()->getCreationTime())
            ->format("%im:%ss");
    }

    /**
     * @param float $memoryPeakUsage
     * @return string
     */
    private function formatMemoryPeakUsage(float $memoryPeakUsage): string
    {
        return sprintf('%05.2f MB', $memoryPeakUsage);
    }
}
