<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Compatibility\Output;

use DateTime;
use Exception;
use JsonException;
use Sut\Domain\Compatibility\Data\AnalysisResult;
use Sut\Domain\Compatibility\OutputInterface;
use Sut\Domain\Issue\IssueSerializer;
use Symfony\Component\Console\Style\SymfonyStyle;

class Json implements OutputInterface
{
    /**
     * @var IssueSerializer
     */
    private $issueSerializer;

    /**
     * @var string
     */
    private $defaultOutputPath;

    /**
     * @param IssueSerializer $issueSerializer
     * @param string $defaultOutputPath
     */
    public function __construct(IssueSerializer $issueSerializer, string $defaultOutputPath)
    {
        $this->issueSerializer = $issueSerializer;
        $this->defaultOutputPath = $defaultOutputPath;
    }

    /**
     * @param SymfonyStyle $output
     * @param AnalysisResult $analysisResult
     * @throws JsonException
     */
    public function write(
        SymfonyStyle $output,
        AnalysisResult $analysisResult
    ): void {
        $outputPath = $analysisResult->getAnalysisRequest()->getJsonOutputPath()
            ?? sprintf($this->defaultOutputPath, (new DateTime())->format('d_M_Y_H:i'));

        $content =  json_encode([
            'issues' => $this->issueSerializer->serializeIssues($analysisResult->getSummary()->getFilteredIssues()),
            'stats' => [
                'currentVersion' => $analysisResult->getAnalysisRequest()->getCurrentVersion(),
                'targetVersion' => $analysisResult->getAnalysisRequest()->getComingVersion(),
                'modulesNumber' => count($analysisResult->getModules()),
                'incompatibleModulesNumber' => $analysisResult->getIncompatibleModulesNumber(),
                'filesNumber' => $analysisResult->getTotalNumberOfFiles(),
                'incompatibleFilesNumber' => count($analysisResult->getIncompatibleFileLevels()),
                'critical' => count($analysisResult->getSummary()->getCriticals()),
                'errors' => count($analysisResult->getSummary()->getErrors()),
                'warnings' => count($analysisResult->getSummary()->getWarnings()),
                'memoryPeakUsage' => $analysisResult->getMemoryPeakUsage()
            ]
        ], JSON_THROW_ON_ERROR | JSON_PRETTY_PRINT);

        try {
            (new SaveFile())->execute($outputPath, $content);
        } catch (Exception $exception) {
            $output->writeln("<error>" . $exception->getMessage() . "</error>");
            return;
        }

        $output->writeln(sprintf("<comment>JSON report saved to '%s'</comment>", $outputPath));
    }
}
