<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Etalon;

use FilesystemIterator;
use RecursiveCallbackFilterIterator;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use Traversable;

class GetIterator
{
    /**
     * @var HashFileInterface
     */
    private $hasher;

    /**
     * @var array
     */
    private $excluded;

    /**
     * @param HashFileInterface $hasher
     * @param string[] $excluded
     */
    public function __construct(HashFileInterface $hasher, array $excluded = [])
    {
        $this->hasher = $hasher;
        $this->excluded = $excluded;
    }

    /**
     * Returns a Traversable that allows to iterate through all files and folders contained in the passed
     * path of the file system, returning the hash signature of each file included.
     *
     * @param string $path
     * @param string[] $excludePackages
     * @return Traversable
     */
    public function execute(string $path, array $excludePackages = []): Traversable
    {
        $excluded = array_merge(
            $this->excluded,
            array_map(
                function ($package) {
                    return 'vendor/' . $package;
                },
                $excludePackages
            )
        );
        $filter = function ($file) use ($path, $excluded) {
            $relativePath = str_replace(rtrim($path, '/') . '/', '', $file->getPathname());
            foreach ($excluded as $prefix) {
                if (strpos($relativePath, $prefix) === 0) {
                    return false;
                }
            }
            return true;
        };

        return new HashIterator(
            new RecursiveIteratorIterator(
                new RecursiveCallbackFilterIterator(
                    new RecursiveDirectoryIterator(
                        $path,
                        FilesystemIterator::SKIP_DOTS
                    ),
                    $filter
                ),
                RecursiveIteratorIterator::SELF_FIRST
            ),
            $this->hasher,
            $path
        );
    }
}
