<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Etalon;

use OuterIterator;
use RecursiveIteratorIterator;
use SplFileInfo;

class HashIterator implements OuterIterator
{
    /**
     * @var RecursiveIteratorIterator
     */
    private $iterator;

    /**
     * @var HashFileInterface
     */
    private $hashFile;

    /**
     * @var string
     */
    private $basePath;

    /**
     * @param RecursiveIteratorIterator $iterator
     * @param HashFileInterface $hashFile
     * @param string $basePath
     */
    public function __construct(RecursiveIteratorIterator $iterator, HashFileInterface $hashFile, string $basePath)
    {
        $this->iterator = $iterator;
        $this->hashFile = $hashFile;
        // We normalize base path adding a trailing slash in case it is missing.
        $this->basePath = rtrim($basePath, '/') . '/';
    }

    /**
     * @param SplFileInfo $file
     * @return string
     */
    private function getFilePath(SplFileInfo $file): string
    {
        return $file->getPath() . '/' . $file->getFilename();
    }

    /**
     * @return string
     */
    public function current(): ?string
    {
        if (is_file($this->getFilePath($this->iterator->current()))) {
            return $this->hashFile->execute($this->getFilePath($this->iterator->current()));
        }

        return null;
    }

    /**
     * @inheritdoc
     */
    public function next(): void
    {
        $this->iterator->next();
    }

    /**
     * Keys of iterable are relative paths to files
     *
     * @return string
     */
    public function key(): string
    {
        return str_replace($this->basePath, '', $this->getFilePath($this->iterator->current()));
    }

    /**
     * @return bool
     */
    public function valid(): bool
    {
        return $this->iterator->valid();
    }

    /**
     * @inheritdoc
     */
    public function rewind(): void
    {
        $this->iterator->rewind();
    }

    /**
     * @return RecursiveIteratorIterator
     */
    public function getInnerIterator(): RecursiveIteratorIterator
    {
        return $this->iterator;
    }
}
