<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Issue\DTO;

/**
 * Class Issue is a DTO which holds information of a specific issue found in the scanned Adobe Commerce instance.
 */
class Issue
{
    private const CODE_TO_STRING = [
        IssueLevel::CRITICAL => 'CRITICAL',
        IssueLevel::ERROR => 'ERROR',
        IssueLevel::WARNING => 'WARNING',
    ];

    /**
     * @var int
     */
    private $lineNumber;

    /**
     * @var string
     */
    private $issueType;

    /**
     * @var int
     */
    private $level;

    /**
     * @var string
     */
    private $message;

    /**
     * @var int
     */
    private $code;

    /**
     * @var string
     */
    private $fileName;

    /**
     * @var string
     */
    private $module;

    /**
     * @var string
     */
    private $validationType;

    /**
     * @param int $level
     * @param string $message
     * @param int $code
     * @param int $lineNumber
     * @param string $issueType
     * @param string $fileName
     * @param string $validationType
     * @param string $module
     */
    public function __construct(
        int $level,
        string $message,
        int $code = 0,
        int $lineNumber = 0,
        string $issueType = '',
        string $fileName = '',
        string $validationType = '',
        string $module = ''
    ) {
        $this->level = $level;
        $this->message = $message;
        $this->code = $code;
        $this->lineNumber = $lineNumber;
        $this->issueType = $issueType;
        $this->fileName = $fileName;
        $this->validationType = $validationType;
        $this->module = $module;
    }

    /**
     * @return string
     */
    public function getIssueType(): string
    {
        return $this->issueType;
    }

    /**
     * @return int
     */
    public function getCode(): int
    {
        return $this->code;
    }

    /**
     * @return int
     */
    public function getLineNumber(): int
    {
        return $this->lineNumber;
    }

    /**
     * @return int
     */
    public function getLevel(): int
    {
        return $this->level;
    }

    /**
     * @return string
     */
    public function getMessage(): string
    {
        return $this->message;
    }

    /**
     * @return string
     */
    public function getFileName(): string
    {
        return $this->fileName;
    }

    /**
     * @return string
     */
    public function getValidationType(): string
    {
        return $this->validationType;
    }

    /**
     * @return string
     */
    public function getLevelString(): string
    {
        $levelName = 'WARNING';
        if (array_key_exists($this->getLevel(), self::CODE_TO_STRING)) {
            $levelName = self::CODE_TO_STRING[$this->getLevel()];
        }
        return $levelName;
    }

    /**
     * @return string
     */
    public function getUniqueId(): string
    {
        $message = $this->getMessage();
        $versionPosition = strpos($message, 'on version');
        if ($versionPosition !== false) {
            $message = substr($message, 0, $versionPosition);
        }
        return sha1($this->getCode() . $this->getFileName() . $this->getLineNumber() . $message);
    }

    /**
     * @param string $module
     */
    public function setModule(string $module): void
    {
        $this->module = $module;
    }

    /**
     * @return string
     */
    public function getModule(): string
    {
        return $this->module;
    }
}
