<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
namespace Sut\Domain\Issue;

use InvalidArgumentException;
use Sut\Domain\Issue\DTO\Issue;

class IssueFactory
{
    private const DEFAULT_LINE_NUMBER = 0;
    private const DEFAULT_ERROR_LEVEL = 3;

    private const VALIDATIONS = [
        1 => 'php',
        2 => 'core',
        3 => 'graphql',
        5 => 'php',
        6 => 'php',
        7 => 'db'
    ];

    /**
     * @var mixed
     */
    private $issuesConfig;

    /**
     * @param array $issuesConfig
     */
    public function __construct(array $issuesConfig)
    {
        $this->issuesConfig = $issuesConfig;
    }

    /**
     * @param int $code
     * @param array $params
     * @param int $lineNumber
     * @param string $fileName
     * @return Issue
     * @throws InvalidArgumentException
     */
    public function createIssue(
        int $code,
        array $params = [],
        int $lineNumber = self::DEFAULT_LINE_NUMBER,
        string $fileName = ''
    ): Issue {
        if (isset($this->issuesConfig['issues'][$code])) {
            return new Issue(
                $this->issuesConfig['issues'][$code]['level'] ?? self::DEFAULT_ERROR_LEVEL,
                sprintf($this->issuesConfig['issues'][$code]['message'], ...$params),
                $code,
                $lineNumber,
                '',
                $fileName,
                $this->getValidationTypeByCode($code)
            );
        }
        throw new InvalidArgumentException(sprintf("Issue code %s is not defined in the configuration", $code));
    }

    /**
     * @param string $type
     * @param string $message
     * @param int $line
     * @param string $filename
     * @return Issue
     */
    public function createIssueByType(
        string $type,
        string $message,
        int $line = self::DEFAULT_LINE_NUMBER,
        string $filename = ''
    ): Issue {
        $issue = $this->getIssueByType($type);
        return new Issue(
            $issue['level'] ?? self::DEFAULT_ERROR_LEVEL,
            $issue['message'] ?? $message,
            $issue['code'],
            $line,
            $issue['type'],
            $filename,
            $this->getValidationTypeByCode($issue['code'])
        );
    }

    /**
     * @param string $type
     * @return array
     * @throws InvalidArgumentException
     */
    private function getIssueByType(string $type): array
    {
        foreach ($this->issuesConfig['issues'] as $code => $issue) {
            if (!isset($issue['type'])) {
                continue;
            }
            if (strpos($type, $issue['type']) === 0) {
                $issue['code'] = $code;
                $issue['type'] = $type;
                return $issue;
            }
        }
        throw new InvalidArgumentException(sprintf('Issue type %s is not defined in the configuration', $type));
    }

    /**
     * @param int $code
     * @return string
     * @throws InvalidArgumentException
     */
    private function getValidationTypeByCode(int $code): string
    {
        $validationTypeCode = ((string)$code)[0];
        if (isset(self::VALIDATIONS[$validationTypeCode])) {
            return self::VALIDATIONS[$validationTypeCode];
        }

        throw new InvalidArgumentException(sprintf(
            "The issue code %d does not match with any know validation type",
            $code
        ));
    }
}
