<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Magento;

use InvalidArgumentException;
use JsonException;
use RuntimeException;
use Sut\Domain\Magento\DTO\MagentoEdition;
use Sut\Domain\Magento\DTO\MagentoInfo;

class GetProjectInfo
{
    private const COMPOSER_LOCK = 'composer.lock';
    private const COMPOSER_JSON = 'composer.json';

    private const VALID_EDITIONS = [
        'magento/product-community-edition' => 'CE',
        'magento/magento2ce' => 'CE',
        'magento/product-enterprise-edition' => 'EE',
        'magento/magento2ee' => 'EE',
    ];

    /**
     * @param string $projectRootPath
     * @return MagentoInfo
     * @throws JsonException
     * @throws InvalidArgumentException
     */
    public function execute(string $projectRootPath): MagentoInfo
    {
        if (!is_dir($projectRootPath)) {
            throw new InvalidArgumentException($projectRootPath . ' is not a directory');
        }

        $magentoInfo = $this->getFromComposerLock($projectRootPath);

        if ($magentoInfo) {
            return $magentoInfo;
        }

        $magentoInfo = $this->getFromComposerJson($projectRootPath);

        if ($magentoInfo) {
            return $magentoInfo;
        }

        throw new RuntimeException('Cannot identify project version');
    }

    /**
     * @param string $projectRootPath
     * @return MagentoInfo|null
     * @throws JsonException
     */
    private function getFromComposerLock(string $projectRootPath): ?MagentoInfo
    {
        if (!is_file(sprintf("%s/%s", $projectRootPath, self::COMPOSER_LOCK))) {
            return null;
        }

        $composerLock = json_decode(
            file_get_contents(
                sprintf(
                    "%s/%s",
                    $projectRootPath,
                    self::COMPOSER_LOCK
                )
            ),
            true,
            512,
            JSON_THROW_ON_ERROR
        );

        if (!isset($composerLock['packages'])) {
            return null;
        }

        foreach ($composerLock['packages'] as $package) {
            if (!isset($package['name'])) {
                continue;
            }
            if (isset(self::VALID_EDITIONS[$package['name']])) {
                if (empty($package['version'])) {
                    return null;
                }
                return new MagentoInfo(
                    $package['version'],
                    $this->getMagentoEdition($package['name'])
                );
            }
        }
        return null;
    }

    /**
     * @param string $projectRootPath
     * @return MagentoInfo|null
     * @throws JsonException
     */
    private function getFromComposerJson(string $projectRootPath): ?MagentoInfo
    {
        if (!is_file(sprintf("%s/%s", $projectRootPath, self::COMPOSER_JSON))) {
            return null;
        }
        $composerJson = json_decode(
            file_get_contents(
                sprintf(
                    "%s/%s",
                    $projectRootPath,
                    self::COMPOSER_JSON
                )
            ),
            true,
            512,
            JSON_THROW_ON_ERROR
        );

        if (!isset($composerJson['name']) || !isset($composerJson['version'])) {
            return null;
        }

        return new MagentoInfo(
            $composerJson['version'],
            $this->getMagentoEdition($composerJson['name'])
        );
    }

    /**
     * @param string $editionName
     * @return MagentoEdition|null
     */
    public function getMagentoEdition(string $editionName) : ?MagentoEdition
    {
        if (!array_key_exists($editionName, self::VALID_EDITIONS)) {
            return null;
        }

        return new MagentoEdition(self::VALID_EDITIONS[$editionName], $editionName);
    }
}
