<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\OutputFormat\Console\Types;

use Sut\Domain\Issue\DTO\Issue;
use Sut\Domain\OutputFormat\Console\OutputFormatInterface;
use Sut\Domain\OutputFormat\Console\OutputFormatService;
use Sut\Domain\Statistics\Statistics;
use Symfony\Component\Console\Style\SymfonyStyle;

class CoreModifications implements OutputFormatInterface
{
    /**
     * @var OutputFormatService
     */
    private $outputFormatService;

    /**
     * @param OutputFormatService $outputFormatService
     */
    public function __construct(OutputFormatService $outputFormatService)
    {
        $this->outputFormatService = $outputFormatService;
    }

    /**
     * @inheritDoc
     */
    public function format(SymfonyStyle $output, array $issues): void
    {
        $this->outputFormatService->addStyles($output);

        $changes = [];

        if (empty($issues)) {
            $output->writeln('No files modified');
            return;
        }

        foreach ($issues as $issue) {
            $changes[] = $this->getIssueLine($issue);
        }
        $output->section('Core files modified:');
        $output->listing($changes);
    }

    /**
     * @param Issue $issue
     * @return string
     */
    private function getIssueLine(Issue $issue): string
    {
        $level = $issue->getLevelString();
        return implode(
            '',
            [
                '<' . strtolower($level) . '>',
                '[' . $level . ']',
                '</' . strtolower($level) . '>',
                '[' . $issue->getCode() . ']',
                ' ' . $issue->getMessage()
            ]
        );
    }

    /**
     * @inheritDoc
     */
    public function showStatistics(SymfonyStyle $output, Statistics $stats) : void
    {
        $output->definitionList(
            ['Running time:' => $stats->getTotalTime()->format("%im:%ss")],
            ['Validated version:' => sprintf(
                "%s %s",
                $stats->getMagentoCurrentVersion() ?: '',
                $stats->getMagentoCurrentEdition() ?: ''
            )
            ],
            ['Core files checked:' => number_format(
                $stats->getTotalCoreCheckedFiles(),
                0,
                ',',
                '.'
            )
            ],
            ['Core files modified:' => number_format(
                $stats->getTotalCoreModifiedFiles(),
                0,
                ',',
                '.'
            )
            ],
            [
                '% of files modified:' => sprintf(
                    "%.2f%%",
                    $stats->getPercentageOfCoreModifiedFiles()
                )
            ],
        );
    }
}
