<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\OutputFormat\Console\Types;

use Sut\Domain\Issue\DTO\Issue;
use Sut\Domain\OutputFormat\Console\OutputFormatInterface;
use Sut\Domain\OutputFormat\Console\OutputFormatService;
use Sut\Domain\Statistics\Statistics;
use Symfony\Component\Console\Helper\TableSeparator;
use Symfony\Component\Console\Style\SymfonyStyle;

class GraphQL implements OutputFormatInterface
{
    /**
     * @var OutputFormatService
     */
    private $outputFormatService;

    /**
     * @param OutputFormatService $outputFormatService
     */
    public function __construct(OutputFormatService $outputFormatService)
    {
        $this->outputFormatService = $outputFormatService;
    }

    /**
     * @inheritDoc
     */
    public function format(SymfonyStyle $output, array $issues): void
    {
        $this->outputFormatService->addStyles($output);

        $output->writeln('Looking for GraphQL Schema problems...');

        if (empty($issues)) {
            $output->writeln('No schema problems found');
            return;
        }

        $issuesList = [];
        foreach ($issues as $issue) {
            $issuesList[] = $this->getIssueLine($issue);
        }

        $output->listing($issuesList);
    }

    /**
     * @param Issue $issue
     * @return string
     */
    private function getIssueLine(Issue $issue): string
    {
        $level = $issue->getLevelString();
        return implode(
            '',
            [
                '<' . strtolower($level) . '>',
                '[' . $level . ']',
                '</' . strtolower($level) . '>',
                '[' . $issue->getCode() . ']',
                ' ' . $issue->getIssueType() . ':',
                ' ' . $issue->getMessage()
            ]
        );
    }

    /**
     * @inheritDoc
     */
    public function showStatistics(SymfonyStyle $output, Statistics $stats) : void
    {
        $messages = [];

        $messages[] = sprintf('Running time: %s', $stats->getTotalTime()->format('%H:%I:%s'));
        $messages[] = new TableSeparator();
        if ($stats->getNumberOfCriticals('graphql') > 0) {
            $messages[] = sprintf('Total GraphQL critical errors found: %d', $stats->getNumberOfCriticals('graphql'));
            foreach ($stats->getCustom()['critical']['issues'] as $error => $amount) {
                $messages[] = "$error: $amount";
            }
        }
        $messages[] = new TableSeparator();
        if ($stats->getNumberOfWarnings('graphql') > 0) {
            $messages[] = sprintf('Total GraphQL warnings found: %d', $stats->getNumberOfWarnings('graphql'));
            foreach ($stats->getCustom()['warnings']['issues'] as $warning => $amount) {
                $messages[] = "$warning: $amount";
            }
        }
        $output->definitionList(
            ...$messages
        );
    }
}
