<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Domain\Tracking\Track;

use Sut\Domain\Tracking\Data\TrackInterface;

class UpdateCheckData implements TrackInterface
{
    /**
     * @var int
     */
    private $executionTime;

    /**
     * @var string
     */
    private $versionFrom;

    /**
     * @var string
     */
    private $versionTo;

    /**
     * @var int
     */
    private $phpErrors;

    /**
     * @var int
     */
    private $phpWarnings;

    /**
     * @var int
     */
    private $phpCritical;

    /**
     * @var bool
     */
    private $success;

    /**
     * @var string|null
     */
    private $context;

    /**
     * @var float
     */
    private $memoryPeakUsage;

    /**
     * @param int $executionTime
     * @param string $versionFrom
     * @param string $versionTo
     * @param float $memoryPeakUsage
     * @param int $phpErrors
     * @param int $phpWarnings
     * @param int $phpCritical
     * @param bool $success
     * @param string|null $context
     */
    public function __construct(
        int $executionTime,
        string $versionFrom,
        string $versionTo,
        float $memoryPeakUsage,
        int $phpErrors = 0,
        int $phpWarnings = 0,
        int $phpCritical = 0,
        bool $success = true,
        ?string $context = null
    ) {
        $this->executionTime = $executionTime;
        $this->versionFrom = $versionFrom;
        $this->versionTo = $versionTo;
        $this->memoryPeakUsage = $memoryPeakUsage;
        $this->phpErrors = $phpErrors;
        $this->phpWarnings = $phpWarnings;
        $this->phpCritical = $phpCritical;
        $this->success = $success;
        $this->context = $context;
    }

    /**
     * @inheritdoc
     */
    public function serialize(): string
    {
        return sprintf(
            '
            {
                "Command" : "upgrade:check",
                "ExecutionTime" : %d,
                "VersionFrom" : "%s",
                "VersionTo" : "%s",
                "MemoryPeakUsage": %f,
                "PhpErrors" : %d,
                "PhpWarnings" : %d,
                "PhpCritical" : %d,
                "Success" : %s,
                "Context": "%s"
            }
            ',
            $this->executionTime,
            $this->versionFrom,
            $this->versionTo,
            $this->memoryPeakUsage,
            $this->phpErrors,
            $this->phpWarnings,
            $this->phpCritical,
            $this->success ? 'true' : 'false',
            $this->context ?? ''
        );
    }
}
