<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Infrastructure\Entrypoint\Command;

use Exception;
use RuntimeException;
use Sut\Domain\Compatibility\Data\AnalysisRequestFactory;
use Sut\Domain\Compatibility\GetIssues;
use Sut\Domain\Compatibility\Index;
use Magento\Mray\MagentoApiIndex\Api;
use Sut\Domain\Compatibility\Output;
use Sut\Domain\Issue\DTO\IssueLevel;
use Sut\Domain\Magento\GetProjectInfo;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class CheckCommand extends Command
{
    private const STRING_TO_CODE = [
        'CRITICAL' => IssueLevel::CRITICAL,
        'ERROR' => IssueLevel::ERROR,
        'WARNING' => IssueLevel::WARNING,
    ];

    private const COMING_VERSION_OPTION = 'coming-version';
    private const CURRENT_VERSION_OPTION = 'current-version';
    private const MIN_ISSUE_LEVEL_OPTION = 'min-issue-level';
    private const IGNORE_CURRENT_VERSION_COMPATIBILITY_ISSUES_OPTION = 'ignore-current-version-compatibility-issues';

    /**
     * @var string
     */
    protected static $defaultName = 'upgrade:check';

    /**
     * @var GetProjectInfo
     */
    private $getProjectInfo;

    /**
     * @var GetIssues
     */
    private $getCompatibilityIssues;

    /**
     * @var Output
     */
    private $output;

    /**
     * @var AnalysisRequestFactory
     */
    private $analysisRequestFactory;

    /**
     * @var Index
     */
    private $index;

    /**
     * @var string
     */
    private $documentationLink;

    /**
     * @param GetProjectInfo $getProjectInfo
     * @param GetIssues $getCompatibilityIssues
     * @param Output $output
     * @param AnalysisRequestFactory $analysisRequestFactory
     * @param Index $index
     * @param string $documentationLink
     */
    public function __construct(
        GetProjectInfo $getProjectInfo,
        GetIssues $getCompatibilityIssues,
        Output $output,
        AnalysisRequestFactory $analysisRequestFactory,
        Index $index,
        string $documentationLink
    ) {
        parent::__construct(self::$defaultName);
        $this->getProjectInfo = $getProjectInfo;
        $this->getCompatibilityIssues = $getCompatibilityIssues;
        $this->output = $output;
        $this->analysisRequestFactory = $analysisRequestFactory;
        $this->index = $index;
        $this->documentationLink = $documentationLink;
    }

    /**
     * @inheritdoc
     */
    protected function configure(): void
    {
        $this
            ->setDescription(
                'The Upgrade Compatibility Tool is a command-line tool that checks an Adobe Commerce '.
                'customized instance against a specific version by analyzing all modules installed in it. '.
                'Returns a list of errors and warnings that must be addressed before upgrading to the latest '.
                'version of Adobe Commerce.'
            )
            ->addArgument('dir', InputArgument::REQUIRED, 'Adobe Commerce installation directory.')
            ->addOption(
                self::CURRENT_VERSION_OPTION,
                'a',
                InputOption::VALUE_OPTIONAL,
                'Current Adobe Commerce version, version of the Adobe Commerce installation will be used if omitted.'
            )
            ->addOption(
                self::COMING_VERSION_OPTION,
                'c',
                InputOption::VALUE_OPTIONAL,
                'Target Adobe Commerce version, latest released version of Adobe Commerce will be used if omitted.' .
                "\n" .
                'Available Adobe Commerce versions: ' . implode(' | ', (new Index(new Api()))->getAvailableVersions())
            )->addOption(
                'json-output-path',
                null,
                InputOption::VALUE_OPTIONAL,
                'Path of the file where the output will be exported in json format'
            )->addOption(
                'html-output-path',
                null,
                InputOption::VALUE_OPTIONAL,
                'Path of the file where the output will be exported in HTML format'
            )->addOption(
                self::MIN_ISSUE_LEVEL_OPTION,
                null,
                InputOption::VALUE_OPTIONAL,
                'Minimal issue level you want to see in the report (warning, error or critical).',
                'warning'
            )->addOption(
                self::IGNORE_CURRENT_VERSION_COMPATIBILITY_ISSUES_OPTION,
                'i',
                InputOption::VALUE_NONE,
                'Ignore common issues for current and coming version'
            )->addOption(
                'context',
                null,
                InputOption::VALUE_REQUIRED,
                'Execution context. This option is for integration purposes and does not affect the execution result.'
            );
    }

    /**
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $io->writeln([
            'Upgrade compatibility tool',
            '',
            sprintf(
                'Check <options=bold>%s</> for a detailed documentation of the Upgrade Compatibility Tool.',
                $this->documentationLink
            ),
            ''
        ]);

        try {
            $minLevel = $this->getMinIssueLevel($input);
            $currentVersion = $this->getCurrentVersion($input);
            $comingVersion = $this->getComingVersion($input);
        } catch (Exception $exception) {
            $io->getErrorStyle()->writeln($exception->getMessage());
            return Command::FAILURE;
        }

        if ($input->getOption(self::COMING_VERSION_OPTION) === null) {
            $io->writeln([
                sprintf(
                    '<comment>No target version specified, UCT will run its checks using ' .
                    'Magento API version %s</comment>',
                    $comingVersion
                ),
                ''
            ]);
        }

        $analysisPath = realpath(rtrim($input->getArgument('dir'), '/'));

        if (!$analysisPath || !file_exists($analysisPath) || !is_dir($analysisPath)) {
            $io->getErrorStyle()->writeln(
                '<error>Requested project path does not exist or is not a directory</error>'
            );
            return Command::FAILURE;
        }

        $output->writeln('Indexing project ' . $analysisPath);
        try {
            $analysisResult = $this->getCompatibilityIssues->execute(
                $io,
                $this->analysisRequestFactory->create(
                    $input,
                    $analysisPath,
                    $currentVersion,
                    $comingVersion,
                    $minLevel
                ),
                $input->getOption(self::IGNORE_CURRENT_VERSION_COMPATIBILITY_ISSUES_OPTION)
            );
            $io->newLine();
            $this->output->write($io, $analysisResult);
        } catch (Exception $exception) {
            $io->getErrorStyle()->writeln('<error>' . $exception->getMessage() . '</error>');
            return Command::FAILURE;
        }
        return Command::SUCCESS;
    }

    /**
     * @param InputInterface $input
     * @return string
     * @throws Exception
     */
    private function getCurrentVersion(InputInterface $input): string
    {
        $currentVersion = $input->getOption(self::CURRENT_VERSION_OPTION);
        if ($currentVersion) {
            return $currentVersion;
        }
        try {
            return $this->getProjectInfo->execute($input->getArgument('dir'))->getVersion();
        } catch (Exception $exception) {
            throw new RuntimeException(
                '<error>Cannot identify the current version of magento project. '
                . 'Please rerun the command specifying the --current-version option</error>'
            );
        }
    }

    /**
     * @param InputInterface $input
     * @return string
     * @throws Exception
     */
    private function getComingVersion(InputInterface $input): string
    {
        $comingVersion = $input->getOption(self::COMING_VERSION_OPTION);
        if ($comingVersion) {
            return $comingVersion;
        }
        $indexedVersions = $this->index->getAvailableVersions();
        if (empty($indexedVersions)) {
            throw new RuntimeException(
                '<error>Cannot identify the coming version of magento project. '
                . 'Please rerun the command specifying the --coming-version option</error>'
            );
        }
        return end($indexedVersions);
    }

    /**
     * @param InputInterface $input
     * @return int
     * @throws Exception
     */
    private function getMinIssueLevel(InputInterface $input): int
    {
        $minIssueLevelInput = strtoupper(trim((string)$input->getOption(self::MIN_ISSUE_LEVEL_OPTION)));
        if (isset(self::STRING_TO_CODE[$minIssueLevelInput])) {
            return self::STRING_TO_CODE[$minIssueLevelInput];
        }
        throw new RuntimeException(
            sprintf(
                '<error>"%s" is not a valid issue level. Valid levels are "critical", "error" and "warning"</error>',
                $input->getOption(self::MIN_ISSUE_LEVEL_OPTION)
            )
        );
    }
}
