<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Infrastructure\Entrypoint\Command;

use DateTime;
use Exception;
use JsonException;
use RuntimeException;
use Sut\Domain\Cmd\CmdServiceInterface;
use Sut\Domain\Compatibility\GetMemoryPeakUsage;
use Sut\Domain\Etalon\GetModifiedFiles;
use Sut\Domain\FileSystem\FileSystemInterface;
use Sut\Domain\Magento\DTO\MagentoInfo;
use Sut\Domain\Magento\GetProjectInfo;
use Sut\Domain\OutputFormat\Console\OutputFormatInterface as ConsoleOutputFormatInterface;
use Sut\Domain\OutputFormat\File\OutputFormatInterface as ExportOutputFormatInterface;
use Sut\Domain\Statistics\Statistics;
use Sut\Domain\Tracking\Track\CoreCodeChangesData;
use Sut\Domain\Tracking\Tracker;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\QuestionHelper;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Output\StreamOutput;
use Symfony\Component\Console\Question\ConfirmationQuestion;
use Symfony\Component\Console\Style\SymfonyStyle;
use Sut\Domain\Etalon\DTO\ModificationCheckResult;

class CoreCodeChangesCommand extends Command
{
    /**
     * Name of the command
     *
     * @var string
     */
    protected static $defaultName = 'core:code:changes';

    private const REPO_URL = 'https://repo.magento.com';

    /**
     * Output formatter
     *
     * @var ConsoleOutputFormatInterface
     */
    private $consoleOutputFormat;

    /**
     * File formatter
     *
     * @var ExportOutputFormatInterface
     */
    private $exportOutputFormat;

    /**
     * @var GetModifiedFiles
     */
    private $getModifiedFiles;

    /**
     * @var GetProjectInfo
     */
    private $getProjectInfo;

    /**
     * @var Tracker
     */
    private $tracker;

    /**
     * @var FileSystemInterface
     */
    private $fileSystem;

    /**
     * @var CmdServiceInterface
     */
    private $cmdService;

    /**
     * @var string
     */
    private $outputPath;

    /**
     * @var string
     */
    private $errorCodesDocumentation;

    /**
     * @var string
     */
    private $documentationLink;

    /**
     * @var GetMemoryPeakUsage
     */
    private $getMemoryPeakUsage;

    /**
     * @param ConsoleOutputFormatInterface $consoleOutputFormat
     * @param ExportOutputFormatInterface $exportOutputFormat
     * @param GetModifiedFiles $getModifiedFiles
     * @param GetProjectInfo $magentoService
     * @param Tracker $tracker
     * @param FileSystemInterface $fileSystem
     * @param CmdServiceInterface $cmdService
     * @param string $outputPath
     * @param string $errorCodesDocumentation
     * @param string $documentationLink
     * @param GetMemoryPeakUsage $getMemoryPeakUsage
     */
    public function __construct(
        ConsoleOutputFormatInterface $consoleOutputFormat,
        ExportOutputFormatInterface $exportOutputFormat,
        GetModifiedFiles $getModifiedFiles,
        GetProjectInfo $magentoService,
        Tracker $tracker,
        FileSystemInterface $fileSystem,
        CmdServiceInterface $cmdService,
        string $outputPath,
        string $errorCodesDocumentation,
        string $documentationLink,
        GetMemoryPeakUsage $getMemoryPeakUsage
    ) {
        parent::__construct(self::$defaultName);
        $this->consoleOutputFormat = $consoleOutputFormat;
        $this->exportOutputFormat = $exportOutputFormat;
        $this->getModifiedFiles = $getModifiedFiles;
        $this->getProjectInfo = $magentoService;
        $this->tracker = $tracker;
        $this->fileSystem = $fileSystem;
        $this->cmdService = $cmdService;
        $this->outputPath = sprintf($outputPath, (new DateTime())->format('d_M_Y_h:i'));
        $this->errorCodesDocumentation = $errorCodesDocumentation;
        $this->documentationLink = $documentationLink;
        $this->getMemoryPeakUsage = $getMemoryPeakUsage;
    }

    /**
     * @inheritdoc
     */
    protected function configure(): void
    {
        $this->setDescription(
            'The Upgrade Compatibility Tool is a command-line tool that checks a Adobe Commerce instance against ' .
            'a specific version by analyzing all the non-Adobe Commerce modules installed in it. Returns a list of ' .
            'errors and warnings that you must address before upgrading to a new version of Adobe Commerce code.'
        )->addArgument(
            'dir',
            InputArgument::REQUIRED,
            'Adobe Commerce installation directory.'
        )->addArgument(
            'vanilla-dir',
            InputArgument::OPTIONAL,
            'Adobe Commerce vanilla installation directory.'
        )->addOption(
            'output',
            'o',
            InputOption::VALUE_OPTIONAL,
            "Path of the file where the output will be exported (Json Format)"
        );
    }

    /**
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $io->writeln(
            [
                'Upgrade compatibility tool - Core Code Changes',
                '',
                sprintf(
                    'Check <options=bold>%s</> for a detailed documentation of the Upgrade Compatibility Tool.',
                    $this->documentationLink
                ),
                ''
            ]
        );

        $outputPath = $input->getOption('output') ?? $this->outputPath;
        $dir = $input->getArgument('dir');
        $vanillaDir = $input->getArgument('vanilla-dir');
        $stats = new Statistics();

        $stats->setStartTime(new DateTime());
        if ($outputPath) {
            $this->fileSystem->createDirectory($outputPath);
        }

        try {
            $magentoInfo = $this->getProjectInfo->execute($dir);
        } catch (Exception $exception) {
            $magentoInfo = new MagentoInfo('');
        }

        if (empty($vanillaDir) && $magentoInfo->getEdition()) {
            $vanillaDir = __DIR__ . '/../../../../var/vanilla/' . $magentoInfo->getVersion();
            if (!$this->downloadVanilla($input, $output, $vanillaDir, $magentoInfo)) {
                $io->getErrorStyle()->writeln('<error>No vanilla Adobe Commerce available for comparison</error>');
                return Command::FAILURE;
            }
        }
        $output->writeln(
            sprintf(
                'Analyzing changes in core code using Adobe Commerce at %s as vanilla reference...',
                $vanillaDir
            )
        );

        $modifiedFiles = $this->getModifiedFiles->execute($dir, $vanillaDir);

        $stats->setEndTime(new DateTime());
        $stats->setTotalCoreCheckedFiles($modifiedFiles->getTotalCheckedFiles());
        $stats->setTotalCoreModifiedFiles($modifiedFiles->getTotalFilesModified());
        $version = $magentoInfo->getVersion() ?? '';
        $edition = $magentoInfo->getEdition() ? $magentoInfo->getEdition()->getEdition() : '';
        $stats->setMagentoCurrentVersion($version);
        $stats->setMagentoCurrentEdition($edition);

        $this->consoleOutputFormat->format(
            $io,
            $modifiedFiles->getIssues()
        );

        if (!empty($modifiedFiles->getIssues())) {
            $output->writeln(sprintf(
                '<comment>Check %s for a detailed list of Upgrade Compatibility Tool errors.</comment>',
                $this->errorCodesDocumentation
            ));
        }

        $stats->setMemoryPeakUsage($this->getMemoryPeakUsage->execute());
        $this->consoleOutputFormat->showStatistics(
            $io,
            $stats
        );

        $this->export($outputPath, $modifiedFiles, $stats);

        $this->trackInformation($stats);

        return Command::SUCCESS;
    }

    /**
     * @param Statistics $stats
     */
    private function trackInformation(Statistics $stats): void
    {
        $executionTime = $stats->getTotalTime();
        $timeInSeconds = $executionTime->s + $executionTime->m * 60 + $executionTime->h * 3600;

        $this->tracker->sendData(
            new CoreCodeChangesData(
                $this->getName(),
                $timeInSeconds,
                $stats->getTotalCoreModifiedFiles(),
                $stats->getPercentageOfCoreModifiedFiles() ?? -1.0,
                $this->getMemoryPeakUsage->execute(),
                true
            )
        );
    }

    /**
     * @param string|null $outputPath
     * @param ModificationCheckResult $modifiedFiles
     * @param Statistics $stats
     * @throws JsonException
     */
    protected function export(?string $outputPath, ModificationCheckResult $modifiedFiles, Statistics $stats): void
    {
        if ($outputPath) {
            $this->exportOutputFormat->export(
                new StreamOutput($this->fileSystem->openFile($outputPath)),
                $modifiedFiles->getIssues(),
                $stats
            );
        }
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @param string $vanillaDir
     * @param MagentoInfo $magentoInfo
     * @return bool
     * @throws RuntimeException
     */
    private function downloadVanilla(
        InputInterface $input,
        OutputInterface $output,
        string $vanillaDir,
        MagentoInfo $magentoInfo
    ): bool {
        if (is_dir($vanillaDir)) {
            $output->writeln(
                sprintf(
                    'Using previously downloaded Adobe Commerce %s %s from %s',
                    $magentoInfo->getEdition(),
                    $magentoInfo->getVersion(),
                    $vanillaDir
                )
            );
            return true;
        }
        if (!$magentoInfo->getEdition()) {
            return false;
        }
        if ($this->confirmDownload($input, $output, $magentoInfo->getVersion())) {
            $this->fileSystem->createDirectory($vanillaDir, true);
            $result = $this->cmdService->execCommand(
                sprintf(
                    'composer create-project --repository-url=%s %s=%s %s',
                    self::REPO_URL,
                    $magentoInfo->getEdition()->getComposerEditionName(),
                    $magentoInfo->getVersion(),
                    $vanillaDir
                )
            );
            return $result->getExitCode() === 0;
        }
        return false;
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @param string $currentVersion
     * @return bool
     */
    private function confirmDownload(InputInterface $input, OutputInterface $output, string $currentVersion): bool
    {
        $helper = $this->getHelper('question');
        /** @var QuestionHelper $helper */
        return $helper->ask(
            $input,
            $output,
            new ConfirmationQuestion(
                sprintf(
                    'Vanilla installation parameter is empty. ' .
                    'Do you want to download a vanilla Adobe Commerce %s (Y/n)',
                    $currentVersion
                )
            )
        );
    }
}
