<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Infrastructure\Entrypoint\Command;

use InvalidArgumentException;
use Magento\Mray\MagentoApiIndex\Api\GetIndexedVersions;
use Magento\Mray\MagentoApiIndex\Model\CompareDbSchemas;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class DbSchemaDiffCommand extends Command
{
    private const ARGUMENT_CURRENT_VERSION = 'current-version';
    private const ARGUMENT_TARGET_VERSION = 'target-version';

    /**
     * @var string
     */
    protected static $defaultName = 'dbschema:diff';

    /**
     * @var string
     */
    private $documentationLink;

    /**
     * @var string[]
     */
    private $availableVersions;

    /**
     * @param string $documentationLink
     */
    public function __construct(
        string $documentationLink
    ) {
        parent::__construct(self::$defaultName);
        $this->documentationLink = $documentationLink;
    }

    /**
     * @inheritdoc
     */
    protected function configure()
    {
        $this->setDescription(
            'Allow to list Adobe Commerce DB schema differences between two selected versions.'
            . "\n" . 'Available versions: ' . implode(' | ', $this->getAvailableVersions())
        );
        $this->addArgument(
            self::ARGUMENT_CURRENT_VERSION,
            InputArgument::REQUIRED,
            'current version (e.g. 2.3.2).'
        );
        $this->addArgument(
            self::ARGUMENT_TARGET_VERSION,
            InputArgument::REQUIRED,
            'target version (e.g. 2.4.5).'
        );
    }

    /**
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $io->writeln([
            'Upgrade compatibility tool',
            '',
            sprintf(
                'Check <options=bold>%s</> for a detailed documentation of the Upgrade Compatibility Tool.',
                $this->documentationLink
            ),
            ''
        ]);

        $versionBase = $input->getArgument(self::ARGUMENT_CURRENT_VERSION);
        $versionToDiff = $input->getArgument(self::ARGUMENT_TARGET_VERSION);

        $this->validateVersions($versionBase, $versionToDiff);

        $io->writeln([
            sprintf(
                '<options=bold>DB schema differences between versions %s and %s:</>',
                $versionBase,
                $versionToDiff
            ),
            ''
        ]);

        $messages = (new CompareDbSchemas())->execute($versionBase, $versionToDiff);

        foreach ($messages as $message) {
            $output->writeln(ucfirst($message));
        }

        $io->writeln([
            '',
            sprintf(
                '<options=bold>Total detected differences between version %s and %s: %s</>',
                $versionBase,
                $versionToDiff,
                count($messages)
            )
        ]);
        return 0;
    }

    /**
     * Ensure the provided versions are different and available in the index
     *
     * @param string $baseVersion
     * @param string $targetVersion
     */
    private function validateVersions(string $baseVersion, string $targetVersion): void
    {
        $incorrectVersion = '';

        if ($baseVersion === $targetVersion) {
            throw new InvalidArgumentException('Cannot compare a version to itself.');
        }

        if (!in_array($baseVersion, $this->getAvailableVersions(), true)) {
            $incorrectVersion = $baseVersion;
        }

        if (!in_array($targetVersion, $this->getAvailableVersions(), true)) {
            $incorrectVersion = $targetVersion;
        }

        if ($incorrectVersion) {
            throw new InvalidArgumentException(
                sprintf(
                    '%s version is not available.\nAvailable versions: %s',
                    $incorrectVersion,
                    implode(' | ', $this->getAvailableVersions())
                )
            );
        }
    }

    /**
     * Retrieve indexed versions available for comparison
     *
     * @return array
     */
    private function getAvailableVersions(): array
    {
        if (!$this->availableVersions) {
            $this->availableVersions = (new GetIndexedVersions())->execute();
        }
        return $this->availableVersions;
    }
}
