<?php
/**
 * Copyright 2021 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Sut\Infrastructure\Entrypoint\Command;

use Exception;
use Sut\Domain\PhpCbf\RefactorIssues;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class RefactorCommand extends Command
{
    /**
     * @var string
     */
    protected static $defaultName = 'refactor';

    /**
     * @var RefactorIssues
     */
    private $refactorIssues;

    /**
     * @var string
     */
    private $documentationLink;

    /**
     * @param RefactorIssues $refactorIssues
     * @param string $documentationLink
     */
    public function __construct(
        RefactorIssues $refactorIssues,
        string $documentationLink
    ) {
        parent::__construct(self::$defaultName);
        $this->refactorIssues = $refactorIssues;
        $this->documentationLink = $documentationLink;
    }

    /**
     * @inheritdoc
     */
    protected function configure(): void
    {
        $this->setDescription(
            'Resolves the issues that can be fixed automatically. The code in the path ' .
            'provided will be updated.'
        )->addArgument(
            'path',
            InputArgument::REQUIRED,
            'Path to resolve issues in.'
        );
    }

    /**
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        $io->writeln([
            sprintf(
                'Check <options=bold>%s</> for a detailed documentation of the Upgrade Compatibility Tool.',
                $this->documentationLink
            )
        ]);

        $projectRoot = realpath(rtrim($input->getArgument('path'), '/'));
        try {
            $refactorOutput = $this->refactorIssues->execute($projectRoot);
            $io->newLine();
            $output->writeln($refactorOutput);
        } catch (Exception $exception) {
            $io->getErrorStyle()->writeln('<error>' . $exception->getMessage() . '</error>');
            return Command::FAILURE;
        }
        return Command::SUCCESS;
    }
}
