<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Integration;

use Composer\Composer;
use Composer\EventDispatcher\EventDispatcher;
use Composer\Factory;
use Composer\IO\ConsoleIO;
use Composer\Package\PackageInterface;
use Composer\Package\RootPackage;
use Composer\Repository\RepositoryFactory;
use Composer\Repository\RepositoryManager;
use Magento\Mray\Composer\PackagesInformation;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Console\Helper\HelperSet;
use Symfony\Component\Console\Input\StringInput;
use Symfony\Component\Console\Output\ConsoleOutput;

class PackagesInformationTest extends TestCase
{
    /**
     * @var RepositoryManager
     */
    private $repositoryManager;

    private const VERSION = '1';

    public function setUp(): void
    {
        $input = new StringInput('test');
        $output = new ConsoleOutput();
        $io = new ConsoleIO($input, $output, new HelperSet());
        $config = Factory::createConfig($io);
        $httpDownloader = Factory::createHttpDownloader($io, $config);
        $composer = new Composer();
        $composer->setPackage(new RootPackage('magento', self::VERSION, self::VERSION));
        $eventDispatcher = new EventDispatcher($composer, $io);

        $this->repositoryManager = RepositoryFactory::manager($io, $config, $httpDownloader, $eventDispatcher);

        foreach (RepositoryFactory::defaultRepos($io, $config, $this->repositoryManager) as $r) {
            $this->repositoryManager->addRepository($r);
        };
    }

    /**
     * @dataProvider packagesVersionsProvider
     * @param        array  $packages
     * @param        string $stability
     * @param        array  $expected
     * @return       void
     */
    public function testExecute(array $packages, string $stability, bool $allVersions, array $expected)
    {
        $packagesInformation = new PackagesInformation();
        /**
         * @var $result PackageInterface[]
         */
        $result = $packagesInformation->execute($this->repositoryManager, $packages, $stability, $allVersions);
        $versions = [];
        foreach ($result as $package) {
            $versions[] = $package->getPrettyVersion();
        }
        self::assertEquals($expected, $versions);
    }

    /**
     * @return array[]
     */
    public function packagesVersionsProvider(): array
    {
        return [
            [
                ['psr/container:^1'],
                'stable',
                true,
                [
                    '1.1.2',
                    '1.1.1',
                    '1.1.0',
                    '1.0.0',
                ],
            ],
            [
                ['psr/container:^1'],
                'dev',
                true,
                [
                    '1.1.2',
                    '1.1.1',
                    '1.1.0',
                    '1.0.0',
                    '1.x-dev',
                    '1.1.x-dev',
                ],
            ],
            [
                ['psr/container:^1.1'],
                'stable',
                true,

                [
                    '1.1.2',
                    '1.1.1',
                    '1.1.0',
                ],
            ],
            [
                ['sebastian/money'],
                'stable',
                true,
                [
                    'v1.6.2',
                    'v1.6.1',
                    'v1.6.0',
                    'v1.5.1',
                    'v1.5.0',
                    'v1.4.0',
                    'v1.3.1',
                    'v1.3.0',
                    'v1.2.0',
                    'v1.1.0',
                    'v1.0.0',
                ],
            ],            [
                ['sebastian/money'],
                'stable',
                false,
                [
                    'v1.6.2',
                ],
            ],
        ];
    }
}
