<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Integration;

use Magento\Mray\Api\CustomProjectIndex;
use PHPUnit\Framework\TestCase;

class CustomProjectIndexTest extends TestCase
{
    /**
     * @var CustomProjectIndex
     */
    private $index;

    public function setUp(): void
    {
        $this->index = new CustomProjectIndex();
    }

    /**
     * @param string $path
     * @param string $version
     * @param array $modules
     * @dataProvider projectsProvider
     */
    public function testExecute(
        string $path,
        string $version,
        array $modules
    ) {
        $result = $this->index->execute($path, $version);

        $this->assertEquals($version, $result->getVersion());
        $this->assertEquals($modules, $this->getImportantData($result->getCustomizations()['modules']));
    }

    /**
     * @param array $modules
     * @return array
     */
    private function getImportantData(array $modules): array
    {
        $data = [];
        foreach ($modules as $module) {
            foreach ($module['dependencies'] as $reference => $value) {
                $data[$module['name']][] = $reference;
            }
        }
        return $data;
    }

    /**
     * @return array[]
     */
    public function projectsProvider(): array
    {
        return [
            [
                __DIR__ . '/_files/customProjects/project5/Vendor/ExtendedModule',
                '2.4.0',
                [
                    'Vendor_ExtendedModule' => [
                        'Magento\Cms\Model\Page',
                        'Magento\Cms\Model\Page::_construct',
                        'Magento\Cms\Model\Page::noRoutePage',
                        'Magento\Catalog\Api\ProductRepositoryInterface',
                        'Magento\Catalog\Api\ProductRepositoryInterface::get',
                        'Magento\Catalog\Api\ProductRepositoryInterface::getById',
                        'Magento\CatalogSearch\Model\Indexer\Fulltext\Action\Full::rebuildStoreIndex'
                    ]
                ],
            ],
            [
                __DIR__ . '/_files/customProjects/project1/Vendor/Module',
                '2.4.0',
                [
                    'Vendor_Module' => [
                        'Magento\CatalogRule\Model\Rule\Condition\Product',
                        'Magento\CatalogRule\Model\Rule\Condition\Product::__construct',
                        'Magento\Catalog\Api\ProductRepositoryInterface',
                        'Magento\Catalog\Api\ProductRepositoryInterface::get',
                        'Magento\Catalog\Model\Product'
                    ]
                ],
            ],
            [
                __DIR__ . '/_files/customProjects/project2',
                '2.4.3',
                [
                    'SecondVendor_Module' => [
                        'Dotdigitalgroup\Email\Api\TierPriceFinderInterface',
                        'Dotdigitalgroup\Email\SomeInterface',
                    ],
                    'MagePal_GoogleTagManager' => [
                        'Magento\Framework\DataObject',
                        'Magento\Sales\Model\ResourceModel\Order\CollectionFactoryInterface',
                        'Magento\Framework\Escaper',
                        'Magento\Framework\DataObject::__construct',
                        'Magento\Sales\Model\ResourceModel\Order\CollectionFactoryInterface::create',
                        'Magento\Framework\Escaper::escapeJsQuote',
                        'Magento\Sales\Model\Order',
                        'Magento\Sales\Model\Order::getAllVisibleItems',
                        'Magento\Sales\Model\Order::getIncrementId',
                        'Magento\Sales\Model\Order::getStoreName',
                        'Magento\Sales\Model\Order::getBaseGrandTotal',
                        'Magento\Sales\Model\Order::getBaseSubtotal',
                        'Magento\Sales\Model\Order::getBaseShippingAmount',
                        'Magento\Sales\Model\Order::getTaxAmount',
                        'Magento\Sales\Model\Order::getCouponCode',
                        'Magento\Sales\Model\Order::getDiscountDescription',
                        'Magento\Sales\Model\Order::getDiscountAmount',
                        'Magento\Sales\Model\Order::getShippingDescription',
                        'Magento\Sales\Model\Order::getShippingMethod',
                        'Magento\Sales\Model\Order::getIsVirtual',
                        'Magento\Sales\Model\Order::getCustomerIsGuest',
                        'Magento\Sales\Model\Order::getPayment',
                        'Magento\Sales\Model\Order\Payment::getMethodInstance',
                    ],
                ],
            ],
            [
                __DIR__ . '/_files/customProjects/project1',
                '2.4.0',
                [
                    'Vendor_Module' => [
                        'Magento\CatalogRule\Model\Rule\Condition\Product',
                        'Magento\CatalogRule\Model\Rule\Condition\Product::__construct',
                        'Magento\Catalog\Api\ProductRepositoryInterface',
                        'Magento\Catalog\Api\ProductRepositoryInterface::get',
                        'Magento\Catalog\Model\Product'
                    ],
                    'Vendor_SecondModule' => [
                        'Magento\Cms\Block\SomeBlock',
                        'Magento\Catalog\Model\Product',
                        'Magento\Catalog\Helper\Image',
                        'Magento\Catalog\Helper\Image::MEDIA_TYPE_CONFIG_NODE',
                        'Magento\Catalog\Model\Product::$someProperty'
                    ]
                ],
            ],
            [
                __DIR__ . '/_files/customProjects/',
                '2.4.3',
                [
                    'Vendor_Module' => [
                        'Magento\CatalogRule\Model\Rule\Condition\Product',
                        'Magento\CatalogRule\Model\Rule\Condition\Product::__construct',
                        'Magento\Catalog\Api\ProductRepositoryInterface',
                        'Magento\Catalog\Api\ProductRepositoryInterface::get',
                        'Magento\Catalog\Model\Product'
                    ],
                    'Vendor_SecondModule' => [
                        'Magento\Cms\Block\SomeBlock',
                        'Magento\Catalog\Model\Product',
                        'Magento\Catalog\Helper\Image',
                        'Magento\Catalog\Helper\Image::MEDIA_TYPE_CONFIG_NODE',
                        'Magento\Catalog\Model\Product::$someProperty'
                    ],
                    'SecondVendor_Module' => [
                        'Dotdigitalgroup\Email\Api\TierPriceFinderInterface',
                        'Dotdigitalgroup\Email\SomeInterface',
                    ],
                    'VendorName_ModuleName' => [
                        'Magento\CatalogRule\Model\Product'
                    ],
                    'Vendor_Module2' => [
                        'Magento\Catalog\Api\ProductRepositoryInterface',
                        'Magento\Catalog\Api\ProductRepositoryInterface::get',
                        'Magento\Catalog\Api\ProductRepository'
                    ],
                    'Vendor_SecondModule2' => [
                        'Magento\Cms\Block\SomeBlock',
                        'Magento\Catalog\Model\Product',
                        'Magento\Catalog\Helper\Image',
                        'Magento\Catalog\Helper\Image::MEDIA_TYPE_CONFIG_NODE',
                        'Magento\Catalog\Model\Product::$someProperty'
                    ],
                    'Vendor_ExtendedModule' => [
                        'Magento\Cms\Model\Page',
                        'Magento\Cms\Model\Page::_construct',
                        'Magento\Cms\Model\Page::noRoutePage',
                        'Magento\Catalog\Api\ProductRepositoryInterface',
                        'Magento\Catalog\Api\ProductRepositoryInterface::get',
                        'Magento\Catalog\Api\ProductRepositoryInterface::getById',
                        'Magento\CatalogSearch\Model\Indexer\Fulltext\Action\Full::rebuildStoreIndex'
                    ],
                    'MagePal_GoogleTagManager' => [
                        'Magento\Framework\DataObject',
                        'Magento\Sales\Model\ResourceModel\Order\CollectionFactoryInterface',
                        'Magento\Framework\Escaper',
                        'Magento\Framework\DataObject::__construct',
                        'Magento\Sales\Model\ResourceModel\Order\CollectionFactoryInterface::create',
                        'Magento\Framework\Escaper::escapeJsQuote',
                        'Magento\Sales\Model\Order',
                        'Magento\Sales\Model\Order::getAllVisibleItems',
                        'Magento\Sales\Model\Order::getIncrementId',
                        'Magento\Sales\Model\Order::getStoreName',
                        'Magento\Sales\Model\Order::getBaseGrandTotal',
                        'Magento\Sales\Model\Order::getBaseSubtotal',
                        'Magento\Sales\Model\Order::getBaseShippingAmount',
                        'Magento\Sales\Model\Order::getTaxAmount',
                        'Magento\Sales\Model\Order::getCouponCode',
                        'Magento\Sales\Model\Order::getDiscountDescription',
                        'Magento\Sales\Model\Order::getDiscountAmount',
                        'Magento\Sales\Model\Order::getShippingDescription',
                        'Magento\Sales\Model\Order::getShippingMethod',
                        'Magento\Sales\Model\Order::getIsVirtual',
                        'Magento\Sales\Model\Order::getCustomerIsGuest',
                        'Magento\Sales\Model\Order::getPayment',
                        'Magento\Sales\Model\Order\Payment::getMethodInstance',
                    ],
                    'Vendor_ModuleDbSchema' => [
                        'patch_list',
                        'patch_list.patch_id',
                        'patch_list.patch_name',
                        'patch_list.PRIMARY',
                        'quote',
                        'quote.extra_column',
                        'quote.custom_index',
                    ]
                ],
            ],
            [
                __DIR__ . '/_files/customProjects/project3',
                '2.4.3-p1',
                [
                    'VendorName_ModuleName' => [
                        'Magento\CatalogRule\Model\Product'
                    ]
                ],
            ],
            [
                __DIR__ . '/_files/customProjects2/diXml',
                '2.4.0',
                [
                    'Vendor_Module' => [
                        'Magento\CatalogRule\Model\Rule\Condition\Product',
                        'Magento\CatalogRule\Model\Rule\Condition\Product::__construct',
                        'Magento\Catalog\Api\ProductRepositoryInterface',
                        'Magento\Catalog\Api\ProductRepositoryInterface::get',
                        'Magento\Catalog\Model\Product',
                        'Magento\Framework\Model\ExecuteCommitCallbacks',
                        'Magento\Framework\Api\Search\Document',
                        'Magento\Framework\Api\Search\DocumentInterface',
                        'Magento\Framework\DB\Adapter\AdapterInterface',
                        'Magento\Framework\Data\Argument\Interpreter\Composite',
                    ],
                    'Vendor_SecondModule' => [
                        'Magento\Cms\Block\SomeBlock',
                        'Magento\Catalog\Model\Product',
                        'Magento\Catalog\Helper\Image',
                        'Magento\Catalog\Helper\Image::MEDIA_TYPE_CONFIG_NODE',
                        'Magento\Catalog\Model\Product::$someProperty'
                    ]
                ],
            ],
            [
                __DIR__ . '/_files/3rd-party-dependency-project',
                '2.4.3-p1',
                [],
            ],
            [
                __DIR__ . '/_files/customProjects/project6',
                '2.4.3-p1',
                [
                    'Vendor_ModuleDbSchema' => [
                        'patch_list',
                        'patch_list.patch_id',
                        'patch_list.patch_name',
                        'patch_list.PRIMARY',
                        'quote',
                        'quote.extra_column',
                        'quote.custom_index',
                    ]
                ],
            ],
        ];
    }
}
