<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
namespace Magento\Mray\Integration;

use InvalidArgumentException;
use Magento\Mray\Console\Command\IndexMagentoReleases;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Console\Tester\CommandTester;
use Symfony\Component\Filesystem\Filesystem;

class IndexMagentoReleasesTest extends TestCase
{
    private const OUTPUT_DIR = __DIR__ . '/../../../var/tmp';
    private const INDEX_SUBDIRECTORIES = [
        'api',
        'components',
        'dictionary',
        'phpCode'
    ];

    /**
     * @var CommandTester
     */
    private $commandTester;

    /**
     * @var IndexMagentoReleases
     */
    private $command;

    protected function setUp(): void
    {
        $this->command = new IndexMagentoReleases();
        $this->commandTester = new CommandTester($this->command);
    }

    protected function tearDown(): void
    {
        (new Filesystem())->remove(self::OUTPUT_DIR);
    }

    /**
     * @param string $path
     * @param string[] $expectedOutputParts
     * @dataProvider pathsProvider
     */
    public function testIndexNoIssues(string $path, array $expectedOutputParts): void
    {
        $commandResult = $this->commandTester->execute(
            [
                'dir' => $path,
                'out' => self::OUTPUT_DIR
            ],
        );

        $this->assertEquals(
            $this->command::SUCCESS,
            $commandResult,
            "Index command Failed\n\n" . $this->commandTester->getDisplay()
        );

        $display = $this->commandTester->getDisplay();

        foreach ($expectedOutputParts as $expectedOutputPart) {
            $this->assertStringContainsString($expectedOutputPart, $display);
        }

        $this->assertFileExists(self::OUTPUT_DIR . '/releases.php');

        foreach (self::INDEX_SUBDIRECTORIES as $directory) {
            $this->assertDirectoryExists(self::OUTPUT_DIR . '/' . $directory);
        }
    }

    /**
     * @return array[]
     */
    public function pathsProvider(): array
    {
        return [
            [
                __DIR__ . '/_files/magentoReleases',
                [
                    'Identified project version "2.4.0"',
                    'Identified project version "2.3.7"'
                ]
            ],
            [
                __DIR__ . '/_files/magentoReleases/project3',
                [
                    'Indexing a single project version "2.4.0"'
                ]
            ]
        ];
    }

    public function testIndexNoArgsPassed(): void
    {
        $this->expectException(\RuntimeException::class);
        $this->commandTester->execute([]);
    }

    public function testIndexEmptyDirPassed(): void
    {
        $this->expectException(InvalidArgumentException::class);
        $this->commandTester->execute(
            [
                'dir' => '',
                'out' => self::OUTPUT_DIR
            ],
        );
    }
}
