<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Unit\CodeStructuralElement\Php\NodeRuntimeTypeResolver;

use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Resolver;
use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Factory;
use phpDocumentor\Reflection\Type;
use PhpParser\Node;
use PhpParser\Node\Expr;
use PhpParser\NodeTraverser;
use PhpParser\NodeVisitor\NameResolver;
use PhpParser\ParserFactory;
use PHPUnit\Framework\TestCase;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\ExpressionRuntimeTypeResolver;
use function sprintf;

class ExprTest extends TestCase
{
    /**
     * @return void
     */
    public function testSingleQuoteString()
    {
        $code = <<<'CODE'
        <?php
        'string';
        CODE;
        $type = $this->resolve($code);
        $this->assertEquals(
            'string',
            (string)$type,
            'Single quote string scalar expression is not recognised as a string type.'
        );
    }

    /**
     * @return void
     */
    public function testDoubleQuoteString()
    {
        $code = <<<'CODE'
        <?php
        "string";
        CODE;

        $type = $this->resolve($code);
        $this->assertEquals(
            'string',
            (string)$type,
            'Double quote string scalar expression is not recognised as a string type.'
        );
    }

    /**
     * @return void
     */
    public function testDoubleQuoteStringWithInterpolation()
    {
        $code = <<<'CODE'
        <?php
        "hello $world";
        CODE;

        $type = $this->resolve($code);
        $this->assertEquals(
            'string',
            (string)$type,
            'Double quote string with interpolation scalar expression is not recognised as a string type.'
        );
    }

    /**
     * @return void
     */
    public function testHeredocString()
    {
        $code = <<<'CODE'
        <?php
        <<<HEREDOC
        string
        HEREDOC;
        CODE;

        $type = $this->resolve($code);
        $this->assertEquals(
            'string',
            (string)$type,
            'String declared with Heredoc syntax is not recognised as a string type.'
        );
    }

    /**
     * @return void
     */
    public function testHeredocStringWithInterpolation()
    {
        $code = <<<'CODE'
        <?php
        <<<HEREDOC
        Hello $world
        HEREDOC;
        CODE;

        $type = $this->resolve($code);
        $this->assertEquals(
            'string',
            (string)$type,
            'String declared with Heredoc syntax and interpolation usage is not recognised as a string type.'
        );
    }

    /**
     * @return void
     */
    public function testNowdocString()
    {
        $code = <<<'CODE'
        <?php
        <<<'NOWDOC'
        string
        NOWDOC;
        CODE;

        $type = $this->resolve($code);
        $this->assertEquals(
            'string',
            (string)$type,
            'String declared with Nowdoc syntax is not recognised as a string type.'
        );
    }

    /**
     * @return void
     */
    public function testInt()
    {
        $code = <<<'CODE'
        <?php
        1;
        CODE;

        $type = $this->resolve($code);
        $this->assertEquals(
            'int',
            (string)$type,
            'Integer scalar expression is not recognised as a int type.'
        );
    }

    /**
     * @return void
     */
    public function testFloat()
    {
        $code = <<<'CODE'
        <?php
        1.0;
        CODE;

        $type = $this->resolve($code);
        $this->assertEquals(
            'float',
            (string)$type,
            'Float scalar expression is not recognised as a float type.'
        );
    }

    /**
     * @param string $code
     * @dataProvider boolExpressions
     */
    public function testBool(string $code)
    {
        $type = $this->resolve($code);
        $this->assertEquals(
            'bool',
            (string)$type,
            'Float scalar expression is not recognised as a float type.'
        );
    }

    /**
     * @return string[][]
     */
    public function boolExpressions(): array
    {
        return [
            [
                '<?php true; ',
            ],
            [
                '<?php false; '
            ]
        ];
    }

    /**
     * @param string $const
     * @param string $expectedType
     *
     * @dataProvider phpMagicConstants
     */
    public function testPhpMagicConst(string $const, string $expectedType)
    {
        $code = "<?php $const;";
        $type = $this->resolve($code);

        $this->assertEquals(
            $expectedType,
            (string)$type,
            sprintf('PHP magic const %s should be resolved to type %s but %s given.', $const, $expectedType, $type)
        );
    }

    /**
     * Return map of available PHP Magic constants and their types
     *
     * @see https://www.php.net/manual/en/language.constants.predefined.php
     *
     * @return array
     */
    public function phpMagicConstants(): array
    {
        return [
            '__LINE__' => ['__LINE__', 'int'],
            '__FILE__' => ['__FILE__', 'string'],
            '__DIR__' => ['__DIR__', 'string'],
            '__FUNCTION__' => ['__FUNCTION__', 'string'],
            '__CLASS__' => ['__CLASS__', 'string'],
            '__TRAIT__' => ['__TRAIT__', 'string'],
            '__METHOD__' => ['__METHOD__', 'string'],
            '__NAMESPACE__' => ['__NAMESPACE__', 'string'],
            // ::class, in fact, is not a Magic constant inside PHP implementation
        ];
    }

    /**
     * @param string $cast
     * @param string $expectedType
     *
     * @dataProvider phpTypeCastingRules
     */
    public function testCastExpression(string $cast, string $expectedType)
    {
        $code = <<<CODE
        <?php
        ($cast)\$var;
        CODE;
        $type = $this->resolve($code);
        $this->assertEquals(
            $expectedType,
            (string)$type,
            sprintf('(%s) casting expected to be resolved to %s but %s given.', $cast, $expectedType, $expectedType)
        );
    }

    /**
     * Type cast expression always resolved to known type.
     *
     * @see https://www.php.net/manual/en/language.types.type-juggling.php#language.types.typecasting
     *
     * @return array
     */
    public function phpTypeCastingRules(): array
    {
        return [
            '(int)' => ['int', 'int'],
            '(integer)' => ['integer', 'int'],
            '(bool)' => ['bool', 'bool'],
            '(boolean)' => ['boolean', 'bool'],
            '(float)' => ['float', 'float'],
            '(double)' => ['double', 'float'],
            '(real)' => ['real', 'float'],
            '(string)' => ['string', 'string'],
            '(binary)' => ['binary', 'string'],
            '(array)' => ['array', 'array'],
            '(object)' => ['object', 'object'],
            '(unset)' => ['unset', 'null'],
        ];
    }

    /**
     * @param string $operator
     * @param string $expectedType
     *
     * @dataProvider phpComparisonOperators
     */
    public function testComparisonOperators(string $operator, string $expectedType)
    {
        $code = <<<CODE
        <?php
        \$var1 $operator \$var2;
        CODE;
        $type = $this->resolve($code);
        $this->assertEquals(
            $expectedType,
            (string)$type,
            sprintf('Comparison operator "%s" always lead to %s but %s given.', $operator, $expectedType, $type)
        );
    }

    /**
     * Comparison operators has predefined type of result.
     *
     * @see https://www.php.net/manual/en/language.operators.comparison.php
     *
     * @return array
     */
    public function phpComparisonOperators(): array
    {
        return [
            'Equal' => ['==', 'bool'],
            'Identical' => ['===', 'bool'],
            'Not equal !=' => ['!=', 'bool'],
            'Not equal <>' => ['<>', 'bool'],
            'Not identical' => ['!==', 'bool'],
            'Less than' => ['<', 'bool'],
            'Greater than' => ['>', 'bool'],
            'Less than or equal to' => ['<=', 'bool'],
            'Greater than or equal to' => ['>=', 'bool'],
            'Spaceship' => ['<=>', 'int'],
        ];
    }

    /**
     * @param string $expr
     * @param string $expectedType
     *
     * @dataProvider phpConditionalOperators
     */
    public function testConditionalOperators(string $expr, string $expectedType)
    {
        $code = "<?php $expr;";
        $type = $this->resolveWithChildPreprocessing($code);
        $this->assertEquals(
            $expectedType,
            (string)$type,
            sprintf('Expression "%s" expected to be resolved to %s but %s given.', $expr, $expectedType, $type)
        );
    }

    /**
     * @return \string[][]
     */
    public function phpConditionalOperators(): array
    {
        return [
            'Ternary (mixed)' => ['$cond ? $a : $b', 'mixed'],
            'Ternary (same)' => ['$cond ? 1 : 0', 'int'],
            'Ternary (different)' => ['$cond ? 1 : "no"', 'int|string'],

            'Coalesce (mixed)' => ['$a ?? 1', 'mixed'],
            // Test data not uses variables to avoid complication of type resolution.
            // Some expressions like 1 ?? 2 does not have a sense in real code.
            'Coalesce (same)' => ['1 ?? 2', 'int'],
            'Coalesce (different)' => ['1 ?? 2.0', 'int|float'],
            'Assign coalesce (mixed)' => ['$a ??= 1', 'mixed'],
            // todo: add test cases with known variable type when any variable type resolution implemented
        ];
    }

    /**
     * @param string $operator
     * @param string $exampleExpr
     *
     * @dataProvider phpLogicalOperators
     */
    public function testLogicalOperators(string $operator, string $exampleExpr)
    {
        $code = <<<CODE
        <?php
        $exampleExpr;
        CODE;
        $type = $this->resolve($code);
        $this->assertEquals(
            'bool',
            (string)$type,
            sprintf('Logical operator "%s" always lead to bool but %s given.', $operator, $type)
        );
    }

    /**
     * Logical operators always return bool values.
     *
     * @see https://www.php.net/manual/en/language.operators.logical.php
     *
     * @return array
     */
    public function phpLogicalOperators(): array
    {
        return [
            'Not' => ['&&', '$a && $b'],
            'And (high priority)' => ['&&', '$a && $b'],
            'Or (high priority)' => ['||', '$a || $b'],
            'And (low priority)' => ['and', '$a and $b'],
            'Xor' => ['xor', '$a xor $b'],
            'Or (low priority)' => ['or', '$a or $b'],
        ];
    }

    /**
     * @param string $expr
     * @param string $expectedType
     *
     * @dataProvider phpBitwiseOperators
     */
    public function testBitwiseOperators(string $expr, string $expectedType)
    {
        $code = "<?php $expr;";
        $type = $this->resolveWithChildPreprocessing($code);
        $this->assertEquals(
            $expectedType,
            (string)$type,
            sprintf('Bitwise operation like %s expected %s but %s given.', $expr, $expectedType, $type)
        );
    }

    /**
     * Most bitwise operators always return string if all operands are strings and integer in all other cases.
     * Shift operators always operate with integers.
     *
     * @see https://www.php.net/manual/en/language.operators.bitwise.php
     *
     * @return array
     */
    public function phpBitwiseOperators(): array
    {
        return [
            'And (ints)' => ['1 & 2', 'int'],
            'And (strings)' => ['"a" & "b"', 'string'],
            'And (string with non-string)' => ['"a" & new stdClass', 'int'],
            'And (string and mixed)' => ['"a" & $b', 'int|string'],
            'And (mixed)' => ['$a & $b', 'int|string'],
            'And (mixed with non-mixed and non-string)' => ['$a & []', 'int'],
            'Or' => ['1 | 2', 'int'],
            'Xor' => ['1 ^ 2', 'int'],
            'Not (int)' => ['~1', 'int'],
            'Not (string)' => ['~"abc"', 'string'],
            'Not (mixed)' => ['~$var', 'int|string'],
            'Shift left' => ['$a << $b', 'int'],
            'Shift right' => ['$a >> $b', 'int'],

            'Assignment And' => ['$var &= 1', 'int'],
            'Assignment Or' => ['$var |= 1', 'int'],
            'Assignment Xor' => ['$var ^= 1', 'int'],
            'Assignment Shift Left' => ['$var <<= 1', 'int'],
            'Assignment Shift Right' => ['$var >>= 1', 'int'],
        ];
    }

    /**
     * @param string $expr
     * @param string $expectedType
     *
     * @dataProvider phpArithmeticOperators
     */
    public function testArithmeticOperators(string $expr, string $expectedType)
    {
        $code = "<?php $expr;";
        $type = $this->resolveWithChildPreprocessing($code);
        $this->assertEquals(
            $expectedType,
            (string)$type,
            sprintf('Arithmetic operation like %s expected %s but %s given.', $expr, $expectedType, $type)
        );
    }

    /**
     * @return \string[][]
     */
    public function phpArithmeticOperators(): array
    {
        return [
            'Identity (int)' => ['+1', 'int'],
            'Identity (float)' => ['+1.0', 'float'],
            'Identity (mixed)' => ['+$a', 'int|float'],
            'Identity (array)' => ['+[]', 'void'], // Always PHP Fatal Error

            'Negation (int)' => ['-1', 'int'],
            'Negation (float)' => ['-1.0', 'float'],
            'Negation (mixed)' => ['-$a', 'int|float'],
            'Negation (array)' => ['-[]', 'void'], // Always PHP Fatal Error

            'Addition (ints)' => ['1+2', 'int'],
            'Addition (floats)' => ['1.0+2.0', 'float'],
            'Addition (int and float)' => ['1+2.0', 'float'],
            'Addition (int and mixed)' => ['1+$a', 'int|float'],
            'Addition (float and mixed)' => ['1.0+$a', 'float'],
            'Addition (mixed)' => ['$a+$b', 'int|float'],
            'Array union' => ['[$a] + [$b]', 'array'],
            'Array union with same type' => ['[0] + [1]', 'int[]'],
            'Array union with different type' => ['[0, 1] + ["a", "b"]', 'array'],
            'Array union with mixed type' => ['[0, 1] + [$a, $b]', 'array'],

            'Subtraction (ints)' => ['1-2', 'int'],
            'Subtraction (floats)' => ['1.0-2.0', 'float'],
            'Subtraction (int and float)' => ['1-2.0', 'float'],
            'Subtraction (int and mixed)' => ['1-$a', 'int|float'],
            'Subtraction (float and mixed)' => ['1.0-$a', 'float'],
            'Subtraction (mixed)' => ['$a-$b', 'int|float'],
            'Subtraction (with array)' => ['1-[]', 'void'], // Always PHP Fatal Error

            'Multiplication (ints)' => ['1*2', 'int'],
            'Multiplication (floats)' => ['1.0*2.0', 'float'],
            'Multiplication (int and float)' => ['1*2.0', 'float'],
            'Multiplication (int and mixed)' => ['1*$a', 'int|float'],
            'Multiplication (float and mixed)' => ['1.0*$a', 'float'],
            'Multiplication (mixed)' => ['$a*$b', 'int|float'],
            'Multiplication (with array)' => ['1*[]', 'void'], // Always PHP Fatal Error

            'Division (ints)' => ['2/1', 'int|float'],
            'Division (floats)' => ['2.0/1.0', 'float'],
            'Division (int and float)' => ['2/1.0', 'float'],
            'Division (int and mixed)' => ['2/$a', 'int|float'],
            'Division (float and mixed)' => ['2.0/$a', 'float'],
            'Division (mixed)' => ['$a/$b', 'int|float'],
            'Division (with array)' => ['1/[]', 'void'], // Always PHP Fatal Error

            'Modulo (ints)' => ['4%3', 'int'],
            'Modulo (floats)' => ['4.0%3.0', 'int'],
            'Modulo (int and float)' => ['4%3.0', 'int'],
            'Modulo (int and mixed)' => ['4%$a', 'int'],
            'Modulo (float and mixed)' => ['4.0%$a', 'int'],
            'Modulo (mixed)' => ['$a%$b', 'int'],
            'Modulo (with array)' => ['4%[]', 'void'], // Always PHP Fatal Error

            'Exponentiation (ints)' => ['1**2', 'int'],
            'Exponentiation (floats)' => ['1.0**2.0', 'float'],
            'Exponentiation (int and float)' => ['1**2.0', 'float'],
            'Exponentiation (int and mixed)' => ['1**$a', 'int|float'],
            'Exponentiation (float and mixed)' => ['1.0**$a', 'float'],
            'Exponentiation (mixed)' => ['$a**$b', 'int|float'],
            'Exponentiation (with array)' => ['1**[]', 'void'], // Always PHP Fatal Error
        ];
    }

    /**
     * @param string $operator
     * @param string $exampleExpr
     *
     * @dataProvider phpStringOperators
     */
    public function testStringOperators(string $operator, string $exampleExpr)
    {
        $code = <<<CODE
        <?php
        $exampleExpr;
        CODE;
        $type = $this->resolve($code);
        $this->assertEquals(
            'string',
            (string)$type,
            sprintf('String operator "%s" always lead to string but %s given.', $operator, $type)
        );
    }

    /**
     * String operators always return string values.
     *
     * @see https://www.php.net/manual/en/language.operators.string.php
     *
     * @return array
     */
    public function phpStringOperators(): array
    {
        return [
            'Concat' => ['.', '$a . $b'],
            'Concat Assigment' => ['.=', '$a .= $b'],
        ];
    }

    /**
     * @return void
     */
    public function testTypeOperator()
    {
        $code = <<<'CODE'
        <?php
        $var instanceof stdClass;
        CODE;
        $type = $this->resolve($code);
        $this->assertEquals(
            'bool',
            (string)$type,
            sprintf('Instanceof operator always returns bool value but %s given', $type)
        );
    }

    /**
     * @return void
     */
    public function testIssetConstruct()
    {
        $code = <<<'CODE'
        <?php
        isset($var);
        CODE;
        $type = $this->resolve($code);
        $this->assertEquals(
            'bool',
            (string)$type,
            sprintf('Isset language construction always returns bool value but %s given', $type)
        );
    }

    /**
     * @return void
     */
    public function testEmptyConstruct()
    {
        $code = <<<'CODE'
        <?php
        empty($var);
        CODE;
        $type = $this->resolve($code);
        $this->assertEquals(
            'bool',
            (string)$type,
            sprintf('Empty language construction always returns bool value but %s given', $type)
        );
    }

    /**
     * @return void
     */
    public function testPrintConstruct()
    {
        $code = <<<'CODE'
        <?php
        print('test');
        CODE;
        $type = $this->resolve($code);
        $this->assertEquals(
            'int',
            (string)$type,
            'Print always return 1 so type should be an int but $s given.'
        );
    }

    /**
     * @return void
     */
    public function testBackticks()
    {
        $code = <<<'CODE'
        <?php
        `ls .`;
        CODE;
        $type = $this->resolve($code);
        $this->assertEquals(
            'string|null',
            (string)$type,
            sprintf(
                'Backticks execute shell command and return output or null if error occurred ' .
                'or no output generated but %s given.',
                $type
            )
        );
    }

    /**
     * @param string $expr
     *
     * @dataProvider exitAliases
     */
    public function testExit(string $expr)
    {
        $code = "<?php $expr;";
        $type = $this->resolve($code);
        $this->assertEquals(
            'void',
            (string)$type,
            sprintf('Exit returns nothing but %s given.', $type)
        );
    }

    /**
     * @return string[][]
     */
    public function exitAliases(): array
    {
        return [
            'exit' => ['exit()'],
            'die' => ['die(0)']
        ];
    }

    /**
     * @param string $expr
     *
     * @dataProvider emptyArrayDeclarations
     */
    public function testEmptyArrayDeclaration(string $expr)
    {
        $code = "<?php $expr; ";
        $type = $this->resolve($code);
        $this->assertEquals(
            'array',
            (string)$type,
            sprintf('Array expected but %s given.', $type)
        );
    }

    /**
     * @return string[][]
     */
    public function emptyArrayDeclarations(): array
    {
        return [
            'Old syntax' => ['array()'],
            'New syntax' => ['[]'],
        ];
    }

    /**
     * @param string $expr
     * @param string $expectedType
     *
     * @dataProvider homogenousArrayDeclarations
     */
    public function testHomogenousArrayDeclarations(string $expr, string $expectedType)
    {
        $code = "<?php $expr;";
        $type = $this->resolveWithChildPreprocessing($code);

        $this->assertEquals(
            $expectedType,
            (string)$type,
            sprintf('Array expected but %s given.', $type)
        );
    }

    /**
     * @return string[][]
     */
    public function homogenousArrayDeclarations(): array
    {
        return [
            'Single element' => ['[1]', 'int[]'],
            'Multiple elements' => ['[1, 2]', 'int[]'],
            'Array of arrays' => ['[[], []]', 'array[]'],
            'Array of integer arrays' => ['[[1, 2, 3], [4, 5]]', 'int[][]'],
            'Array of same type objects' => ['[new stdClass, new stdClass]', '\stdClass[]'],
            'Array of different type objects' => ['[new stdClass, new otherClass]', 'array'],
        ];
    }

    /**
     * @param string $expr
     * @param string $expectedType
     *
     * @dataProvider arrayDimFetch
     */
    public function testArrayDimFetch(string $expr, string $expectedType)
    {
        $code = "<?php {$expr}[0];";
        $type = $this->resolveWithChildPreprocessing($code);

        $this->assertEquals(
            $expectedType,
            (string)$type,
            sprintf('Element of array %s expected to be %s but %s given.', $expr, $expectedType, $type)
        );
    }

    /**
     * @return string[][]
     */
    public function arrayDimFetch(): array
    {
        return [
            'Homogenous array' => ['[1,2]', 'int'],
            'Mixed array' => ['[1,false,"a"]', 'mixed'],
            'Unknown types' => ['[$a,$b]', 'mixed'],
        ];
    }

    /**
     * @param $expr
     * @param $expectedType
     *
     * @dataProvider objectInstantiationExpressions
     */
    public function testObjectInstantiation($expr, $expectedType)
    {
        $code = "<?php $expr; ";
        $type = $this->resolve($code);
        $this->assertEquals(
            $expectedType,
            (string)$type,
            sprintf('%s expected but %s given.', $expectedType, $type)
        );
    }

    /**
     * @return string[][]
     */
    public function objectInstantiationExpressions(): array
    {
        return [
            'Explicit type' => ['new stdClass()', '\stdClass'],
            'Anonymous class from base class' => ['new class extends stdClass {}', '\stdClass'],
            'Anonymous class from interfaces' =>
                ['new class implements Countable, SeekableIterator {}', '\Countable|\SeekableIterator'],
            'Anonymous class from base class and interfaces' =>
                ['new class extends stdClass implements Countable {}', '\stdClass|\Countable'],
            'With expression' => ['new $className', 'object'],
            'With self keyword' => ['new self', 'object'],
        ];
    }

    /**
     * @return void
     */
    public function testClone()
    {
        $code = "<?php clone (new stdClass); ";
        $type = $this->resolveWithChildPreprocessing($code);
        $this->assertEquals(
            '\stdClass',
            (string)$type,
            sprintf('\stdClass expected but %s given.', $type)
        );
    }

    /**
     * @return void
     */
    public function testClosures()
    {
        $code = "<?php function () {}; ";
        $type = $this->resolve($code);
        $this->assertEquals(
            'callable',
            (string)$type,
            sprintf('Callable expected but %s given.', $type)
        );
    }

    /**
     * @return void
     */
    public function testArrowFunctions()
    {
        $code = "<?php fn () => 1; ";
        $type = $this->resolve($code);
        $this->assertEquals(
            'callable',
            (string)$type,
            sprintf('Callable expected but %s given.', $type)
        );
    }

    /**
     * @param string $code
     * @param $expectedType
     * @dataProvider matchExamples
     */
    public function testMatch(string $code, $expectedType)
    {
        $type = $this->resolveWithChildPreprocessing($code);
        $this->assertEquals(
            $expectedType,
            (string)$type,
            sprintf('%s expected but %s given.', $expectedType, $type)
        );
    }

    /**
     * @return string[][]
     */
    public function matchExamples(): array
    {
        return [
            [
                <<<'CODE'
                <?php
                match($var) {
                    'foo' => 1,
                    'bar' => 2,
                    default => 3,
                };
                CODE,
                'int'
            ],
            [
                <<<'CODE'
                <?php
                match($var) {
                    'foo' => 1,
                    'bar' => 2,
                    default => 'not found',
                };
                CODE,
                'mixed'
            ],
        ];
    }

    /**
     * @param string $code
     * @return Type
     */
    private function resolve(string $code): Type
    {
        $expr = $this->parse($code);

        $resolver = $this->createResolver();
        return $resolver->resolve($expr);
    }

    /**
     * @param string $code
     * @return Type
     */
    private function resolveWithChildPreprocessing(string $code): Type
    {
        $ast = $this->parse($code);
        $resolver = $this->createResolver();
        $visitor = new ExpressionRuntimeTypeResolver($resolver);

        $nodeTraverser = new NodeTraverser();
        $nodeTraverser->addVisitor($visitor);
        $ast = $nodeTraverser->traverse([$ast]);
        /** @var Node\Expr $expr */
        $expr = $ast[0];
        return $resolver->resolve($expr);
    }

    /**
     * @param string $code
     * @return Expr
     */
    private function parse(string $code): Expr
    {
        $parser = (new ParserFactory())->create(ParserFactory::ONLY_PHP7);
        $ast = $parser->parse($code);

        $traverser = new NodeTraverser();
        $traverser->addVisitor(new NameResolver());
        $ast = $traverser->traverse($ast);

        return $ast[0]->expr;
    }

    /**
     * @return Resolver
     */
    private function createResolver(): Resolver
    {
        return Factory::default()->create();
    }
}
