<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Unit\CodeStructuralElement\Php\NodeVisitor;

use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Factory as TypeResolverFactory;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\ContextSwitcher;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\ExpressionRuntimeTypeResolver;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\StructuralElementsReflector;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\UsageDiscovery;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\VariableDeclarationReflector;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Context\Global_;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Context\Pointer;
use Magento\Mray\CodeStructuralElement\Php\Usage\UsageRegistry;
use PhpParser\Lexer\Emulative as EmulativeLexer;
use PhpParser\NodeTraverser;
use PhpParser\NodeVisitor\NameResolver;
use PhpParser\NodeVisitor\ParentConnectingVisitor;
use PhpParser\ParserFactory;
use PHPUnit\Framework\TestCase;
use function sprintf;

class UsageDiscoveryTest extends TestCase
{
    public function testClassExtended()
    {
        $code = <<<'CODE'
        <?php
        namespace ns1;
        use ns2\C1;
        class C2 extends C1 {}
        CODE;

        $codeUsages = new UsageRegistry();
        $this->discoverUsages($code, $codeUsages);

        $usages = $codeUsages->getUsages();
        $this->assertTrue(isset($usages['ns2\C1']['extend-class']), 'Class extended usage should be detected.');
        $usageInfo = $usages['ns2\C1']['extend-class'][0]->describe();
        $this->assertEquals('ns1\C2', $usageInfo['class'], 'Child class information should be available');
    }

    public function testInterfaceImplemented()
    {
        $code = <<<'CODE'
        <?php
        namespace ns1;
        use ns2\I;
        class C implements I, Serializable {}
        CODE;

        $codeUsages = new UsageRegistry();
        $this->discoverUsages($code, $codeUsages);

        $usages = $codeUsages->getUsages();
        $this->assertTrue(
            isset($usages['ns2\I']['implement-interface']),
            'Interface implemented usage should be detected.'
        );
        $usageInfo = $usages['ns2\I']['implement-interface'][0]->describe();
        $this->assertEquals('ns1\C', $usageInfo['class'], 'Implementing class information should be available');
    }

    public function testInterfaceExtended()
    {
        $code = <<<'CODE'
        <?php
        namespace ns1;
        use ns2\I1;
        interface I2 extends I1, Serializable {}
        CODE;

        $codeUsages = new UsageRegistry();
        $this->discoverUsages($code, $codeUsages);

        $usages = $codeUsages->getUsages();
        $this->assertTrue(isset($usages['ns2\I1']['extend-interface']), 'Interface extended usage should be detected.');
        $usageInfo = $usages['ns2\I1']['extend-interface'][0]->describe();
        $this->assertEquals('ns1\I2', $usageInfo['interface'], 'Extending interface information should be available');
    }

    /**
     * @param string $code
     * @param array $stackInfoExpectations
     * @dataProvider methodCalls
     */
    public function testMethodCall(string $code, array $stackInfoExpectations)
    {
        $codeUsages = new UsageRegistry();
        $this->discoverUsages($code, $codeUsages);

        $usages = $codeUsages->getUsages();
        $this->assertTrue(isset($usages['ns1\C::m']['call-method']), 'Method call usage should be detected.');
        $usageInfo = $usages['ns1\C::m']['call-method'][0]->describe();
        foreach ($stackInfoExpectations as $expectedK => $expectedV) {
            $this->assertEquals($expectedV, $usageInfo[$expectedK], 'Expected declaration stack correctly recognized.');
        }
    }

    public function methodCalls()
    {
        return [
            'from global context' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    public function m() {}
                }
                
                $o = new C();
                $o->m();
                CODE,
                []
            ],
            'from other method' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    public function m() {}
                }
                
                class C2 {
                    public function doSomething() {
                        $o = new C();
                        $o->m();    
                    }
                }
                CODE,
                [
                    'class' => 'ns1\C2',
                    'method' => 'doSomething',
                ]
            ],
            'inherited method' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    public function m() {}
                }
                
                class C2 extends C {
                    public function doSomething() {
                        
                    }
                }
                
                $o = new C2();
                $var = $o->m();
                CODE,
                []
            ],
            'static method' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    public function m() {}
                }
                
                class C2 {
                    public function doSomething() {
                        $o = C::m();    
                    }
                }
                CODE,
                [
                    'class' => 'ns1\C2',
                    'method' => 'doSomething',
                ]
            ],
            'inherited static method through self' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    public function m() {}
                }
                
                class C2 extends C {
                    public function doSomething() {
                        $o = self::m();    
                    }
                }
                CODE,
                [
                    'class' => 'ns1\C2',
                    'method' => 'doSomething',
                ]
            ],
            'parent method' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    public function m() {}
                }
                
                class C2 extends C {
                    public function doSomething() {
                        $o = parent::m();    
                    }
                }
                CODE,
                [
                    'class' => 'ns1\C2',
                    'method' => 'doSomething',
                ]
            ],
        ];
    }

    /**
     * @param string $code
     * @param bool $isFetch
     * @dataProvider propertyOperations
     */
    public function testPropertyFetch(string $code, bool $isFetch)
    {
        $codeUsages = new UsageRegistry();
        $this->discoverUsages($code, $codeUsages);

        $usages = $codeUsages->getUsages();
        if ($isFetch) {
            $this->assertTrue(
                isset($usages['ns1\C::$p']['fetch-property']),
                'Property fetch usage should be detected.'
            );
        } else {
            $this->assertFalse(isset($usages['ns1\C::$p']['fetch-pproperty']), 'Code does not have property fetch.');
        }
    }

    public function propertyOperations()
    {
        return [
            'Property fetch' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    public property $p;
                }

                $o = new C();
                $var = $o->p;
                CODE,
                true
            ],
            'Static property fetch' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    static public property $p;
                }

                $var = C::$p;
                CODE,
                true
            ],
            'Property assignment' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    public property $p;
                }
                
                $o = new C();
                $o->p = 'val';
                CODE,
                false
            ],
            'Static property assignment' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    static public property $p;
                }

                C::$p = 'foo';
                CODE,
                false
            ],
            'Nested property assignment' => [
                <<<'CODE'
                <?php
                namespace ns1;
                class C {
                    public property $p;
                }

                $o = new C();
                $o->p->foo = 'val';
                CODE,
                true
            ],
        ];
    }

    /**
     * @param string $code
     * @param bool $isFetch
     * @dataProvider propertyOperations
     */
    public function testPropertyAssign(string $code, bool $isFetch)
    {
        $codeUsages = new UsageRegistry();
        $this->discoverUsages($code, $codeUsages);

        $usages = $codeUsages->getUsages();
        if ($isFetch) {
            $this->assertFalse(
                isset($usages['ns1\C::$p']['assign-property']),
                'Code does not have property assignment.'
            );
        } else {
            $this->assertTrue(
                isset($usages['ns1\C::$p']['assign-property']),
                'Property assignment usage should be detected.'
            );
        }
    }

    /**
     * @param string $code
     * @param string $expectedConst
     * @dataProvider constFetch
     */
    public function testClassConstantFetch(string $code, string $expectedConst)
    {
        $codeUsages = new UsageRegistry();
        $this->discoverUsages($code, $codeUsages);

        $usages = $codeUsages->getUsages();
        $this->assertTrue(
            isset($usages[$expectedConst]['fetch-const']),
            'Class constant fetch usage should be detected.'
        );
    }

    public function constFetch()
    {
        return [
            'explicit' => [
                <<<'CODE'
                <?php
                namespace ns1;
                use ns2\C;
                
                $var = C::CNST;
                CODE,
                'ns2\C::CNST'
            ],
            'self' => [
                <<<'CODE'
                <?php
                namespace ns1;
                use ns2\C;
                
                class Child extends C
                {
                    public function m() {
                        return self::CNST;
                    }
                }
                CODE,
                'ns1\Child::CNST'
            ],
            'static' => [ // LSB cannot be implemented at this level. treated as self
                <<<'CODE'
                <?php
                namespace ns1;
                use ns2\C;
                
                class Child extends C
                {
                    public function m() {
                        return static::CNST;
                    }
                }
                CODE,
                'ns1\Child::CNST'
            ],
            'parent' => [
                <<<'CODE'
                <?php
                namespace ns1;
                use ns2\C;
                
                class Child extends C
                {
                    public function m() {
                        return parent::CNST;
                    }
                }
                CODE,
                'ns2\C::CNST'
            ],
            'self with full declaration info' => [
                <<<'CODE'
                <?php
                
                class C1 {
                    const CNST = 'foo';
                }
                
                class C2 extends C1
                {
                    public function m() {
                        return self::CNST;
                    }
                }
                CODE,
                'C1::CNST'
            ],
            'parent with full declaration info' => [
                <<<'CODE'
                <?php
                
                class C1 {
                    const CNST = 'foo';
                }
                
                class C2 extends C1
                {
                    public function m() {
                        return parent::CNST;
                    }
                }
                CODE,
                'C1::CNST'
            ],
        ];
    }

    /**
     * @param string $code
     * @param string $target
     * @param string|null $expectedUsageType
     * @dataProvider usageThroughInheritance
     */
    public function testUsageThroughInheritance(string $code, string $target, ?string $expectedUsageType)
    {
        $codeUsages = new UsageRegistry();
        $this->discoverUsages($code, $codeUsages);

        $usages = $codeUsages->getUsages();
        if (isset($expectedUsageType)) {
            $this->assertTrue(
                isset($usages[$target][$expectedUsageType]),
                'Usage through inheritance should be detected.'
            );
        } else {
            $this->assertFalse(
                isset($usages[$target][$expectedUsageType]),
                'No usage though inheritance here.'
            );
        }
    }

    public function usageThroughInheritance()
    {
        return [
            'method override' => [
                <<<'CODE'
                <?php
                
                class C1 {
                    public function m() {}
                }
                
                class C2 extends C1 {
                    public function m() {}
                }
                CODE,
                'C1::m',
                'override-method'
            ],
            'method inheritance' => [
                <<<'CODE'
                <?php
                
                class C1 {
                    public function m() {}
                }
                
                class C2 extends C1 {
                    
                }
                CODE,
                'C1::m',
                'inherit-method'
            ],
            'abstract method implementation' => [
                <<<'CODE'
                <?php
                
                class C1 {
                    abstract public function m() {}
                }
                
                class C2 extends C1 {
                    public function m() {}
                }
                CODE,
                'C1::m',
                'implement-method'
            ],
            'interface method implementation' => [
                <<<'CODE'
                <?php
                
                interface I {
                    public function m() {}
                }
                
                class C implements I {
                    public function m() {}
                }
                CODE,
                'I::m',
                'implement-method'
            ],
            'child-only method' => [
                <<<'CODE'
                <?php
                
                class C1 {
                    
                }
                
                class C2 extends C1 {
                    public function m() {}
                }
                CODE,
                'C2::m',
                null
            ],
        ];
    }

    /**
     * @param string $code
     * @param bool $isReferenceExample
     * @dataProvider directNameReference
     */
    public function testDirectNameReference(string $code, bool $isReferenceExample)
    {
        $codeUsages = new UsageRegistry();
        $this->discoverUsages($code, $codeUsages);

        $usages = $codeUsages->getUsages();
        $this->assertEquals(
            $isReferenceExample,
            isset($usages['ns\C1']['reference']),
            sprintf('Expected class reference %s detected.', $isReferenceExample ? 'is' : 'is not')
        );
    }

    public function directNameReference()
    {
        return [
            'parameter' => [
                <<<'CODE'
                <?php
                
                namespace ns;
                
                use ns2\C;
                
                class C1 {}
                
                class C2 extends C1 {
                    public function m(C1 $o) {}
                }
                CODE,
                true,
            ],
            'instanceof check' => [
                <<<'CODE'
                <?php
                
                namespace ns;
                
                class C1 {}
                
                class C2 {
                    public function m($o) {
                        if ($o instanceof C1) {
                        }
                    }
                }
                CODE,
                true,
            ]
        ];
    }

    private function discoverUsages(string $code, UsageRegistry $registry)
    {
        $phpParser = (new ParserFactory())->create(
            ParserFactory::ONLY_PHP7,
            new EmulativeLexer([
                'usedAttributes' => [
                    'comments', // capture comments
                    'startLine', // capture node start line in file
                    'endLine', // capture node end line in file
                    'startFilePos', // capture position in file to convert to column in start line
                    'endFilePos' // capture position in file to convert to column in end line
                ],
            ])
        );

        $context = new Pointer(new Global_());
        $traverser = new NodeTraverser();

        $traverser->addVisitor(new NameResolver());
        $traverser->addVisitor(new ParentConnectingVisitor());
        $traverser->addVisitor(new StructuralElementsReflector($context));
        $traverser->addVisitor(new ContextSwitcher($context));
        $traverser->addVisitor(new VariableDeclarationReflector($context));
        $traverser->addVisitor(new ExpressionRuntimeTypeResolver(TypeResolverFactory::default()->create(), $context));
        $traverser->addVisitor(new UsageDiscovery($registry));

        $traverser->traverse($phpParser->parse($code) ?? []);
    }
}
