<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Unit\CodeStructuralElement\Php;

use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\StructuralElementRegistrar;
use Magento\Mray\CodeStructuralElement\Php\Parser\Factory as PhpParserFactory;
use Magento\Mray\CodeStructuralElement\Php\PhpFqsen;
use Magento\Mray\CodeStructuralElement\Php\PhpStructuralElement;
use Magento\Mray\CodeStructuralElement\Php\PhpStructuralElementRegistry;
use Magento\Mray\CodeStructuralElement\StructuralElementRegistry;
use PhpParser\NodeTraverser;
use PHPUnit\Framework\TestCase;
use function sprintf;

class PhpStructuralElementRegistryTest extends TestCase
{
    /**
     * @param string $code
     * @param PhpFqsen $expectedFqsen
     * @dataProvider structuralElementsDeclaration
     */
    public function testStructuralElementDeclarationsCaptured(string $code, PhpFqsen $expectedFqsen)
    {
        $seReg = $this->registerStructureElementsFromTheCode($code);
        $this->assertInstanceOf(
            PhpStructuralElement::class,
            $seReg->find($expectedFqsen),
            sprintf('Structural element %s was not recognized and added to registry.', $expectedFqsen)
        );
    }
    public function structuralElementsDeclaration()
    {
        return [
            'Class' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class C {}
                CODE,
                new PhpFqsen\ClassLike('Ns\C'),
            ],
            'Class Method' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class C {
                    /**
                     * @param int $v
                     */
                    public function m($v) {
                        return $v*2;
                    }
                }
                CODE,
                new PhpFqsen\ClassLikeMethod('Ns\C', 'm'),
            ],
            'Class Method From DocBlock' => [
                <<<'CODE'
                <?php
                namespace Ns;
                /**
                 * @method string m(int $v)
                 */
                class C {}
                CODE,
                new PhpFqsen\ClassLikeMethod('Ns\C', 'm'),
            ],
            'Class Property' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class C {
                    public string $p;
                }
                CODE,
                new PhpFqsen\ClassLikeProperty('Ns\C', 'p'),
            ],
            'Class Property From DocBlock' => [
                <<<'CODE'
                <?php
                namespace Ns;
                /**
                 * @property string $p
                 */
                class C {}
                CODE,
                new PhpFqsen\ClassLikeProperty('Ns\C', 'p'),
            ],
            'Class Constant' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class C {
                    const C = 'cConst';
                }
                CODE,
                new PhpFqsen\ClassLikeConstant('Ns\C', 'C'),
            ],
            'Interface' => [
                <<<'CODE'
                <?php
                namespace Ns;
                interface I {}
                CODE,
                new PhpFqsen\ClassLike('Ns\I'),
            ],
            'Interface Method' => [
                <<<'CODE'
                <?php
                namespace Ns;
                interface I {
                    public function m();
                }
                CODE,
                new PhpFqsen\ClassLikeMethod('Ns\I', 'm'),
            ],
            'Interface Constant' => [
                <<<'CODE'
                <?php
                namespace Ns;
                interface I {
                    const C = 'iConst';
                }
                CODE,
                new PhpFqsen\ClassLikeConstant('Ns\I', 'C'),
            ],
        ];
    }

    /**
     * @param string $code
     * @param PhpFqsen $privateElementFqsen
     * @dataProvider privateStructureElements
     */
    public function testPrivateElementsAreIgnored(string $code, PhpFqsen $privateElementFqsen)
    {
        $seReg = $this->registerStructureElementsFromTheCode($code);
        $this->assertNull(
            $seReg->find($privateElementFqsen),
            sprintf('Structural element %s is private and should be ignored.', $privateElementFqsen)
        );
    }
    public function privateStructureElements()
    {
        return [
            'Class Constant' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class C {
                    private const C = 'cConst';
                }
                CODE,
                new PhpFqsen\ClassLikeConstant('Ns\C', 'C'),
            ],
            'Class Property' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class C {
                    private $p;
                }
                CODE,
                new PhpFqsen\ClassLikeProperty('Ns\C', 'p'),
            ],
            'Class Method' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class C {
                    private function m() {}
                }
                CODE,
                new PhpFqsen\ClassLikeMethod('Ns\C', 'm'),
            ]
        ];
    }

    /**
     * @param string $code
     * @param PhpFqsen $expectedFqsen
     * @dataProvider implicitStructuralElements
     */
    public function testImplicitElements(string $code, PhpFqsen $expectedFqsen)
    {
        $seReg = $this->registerStructureElementsFromTheCode($code);
        $this->assertInstanceOf(
            PhpStructuralElement::class,
            $seReg->find($expectedFqsen),
            sprintf('Structural element %s inherited from parent was not added to child element.', $expectedFqsen)
        );
    }
    public function implicitStructuralElements()
    {
        return [
            'Class has method from parent (parent parsed before child)' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class ParentClass {
                    public function m() {}
                }
                class ChildClass extends ParentClass {

                }
                CODE,
                new PhpFqsen\ClassLikeMethod('Ns\ChildClass', 'm'),
            ],
            'Class has method from parent (parent parsed after child)' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class ChildClass extends ParentClass {

                }
                class ParentClass {
                    public function m() {}
                }
                CODE,
                new PhpFqsen\ClassLikeMethod('Ns\ChildClass', 'm'),
            ],
            'Method from interface (interface parsed before class)' => [
                <<<'CODE'
                <?php
                namespace Ns;
                interface I {
                    public function m() {}
                }
                class C implements I {

                }
                CODE,
                new PhpFqsen\ClassLikeMethod('Ns\C', 'm'),
            ],
            'Method from interface (interface parsed after class)' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class C implements I {

                }
                interface I {
                    public function m() {}
                }
                CODE,
                new PhpFqsen\ClassLikeMethod('Ns\C', 'm'),
            ],
            'Multiple inheritance' => [
                <<<'CODE'
                <?php
                namespace Ns;
                class ParentClass implements I {

                }
                class ChildClass extends ParentClass {

                }
                interface I {
                    public function m() {}
                }
                CODE,
                new PhpFqsen\ClassLikeMethod('Ns\ChildClass', 'm'),
            ]
        ];
    }

    private function registerStructureElementsFromTheCode(string $code): StructuralElementRegistry
    {
        $registry = new PhpStructuralElementRegistry();
        $traverser = new NodeTraverser();
        $traverser->addVisitor(new StructuralElementRegistrar($registry));
        $traverser->traverse(PhpParserFactory::create()->parse($code)->getStatements());

        return $registry;
    }
}
