<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Api;

use Magento\Mray\Index\Collection\DbSchema;
use Magento\Mray\Index\Traverser\MagentoDbSchemaDiscovery;
use Magento\Mray\Parse\Xml\XMLStringToAssocArray;
use RuntimeException;
use Magento\Mray\Api\Data\MagentoReleaseIndexResult;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Context\StructuralElementDeclarations;
use Magento\Mray\Index\Collection\Dictionary;
use Magento\Mray\Index\Collection\MagentoApi;
use Magento\Mray\Index\Collection\MagentoComponents;
use Magento\Mray\Index\Collection\PhpCode;
use Magento\Mray\Index\Collection\VirtualTypes;
use Magento\Mray\Index\Scanner\Tuner;
use Magento\Mray\Index\Traverser\MagentoApiDiscovery;
use Magento\Mray\Index\Traverser\MagentoComponentsDiscovery;
use Magento\Mray\Index\Traverser\PhpCodeDiscovery;
use Magento\Mray\Index\Traverser\MagentoVirtualTypesDiscovery;
use Magento\Mray\Index\Traverser\PhpCodeRuntimeTypesEstimation;

class MagentoReleaseIndex
{
    /**
     * @param string $path
     * @return MagentoReleaseIndexResult
     * @throws RuntimeException
     */
    public function execute(string $path): MagentoReleaseIndexResult
    {
        $version = (new GetProjectVersion())->execute($path);
        $phpElementsRegistry = new StructuralElementDeclarations();
        $dictionary = new Dictionary();
        $componentsIndex = new MagentoComponents($dictionary);
        $phpCodeIndex = new PhpCode($dictionary);
        $apiIndex = new MagentoApi($dictionary);
        $virtualTypes = new VirtualTypes();
        $dbSchema = new DbSchema();

        $idx = new MagentoIndexEngine(
            [
                new Tuner\MagentoComposerDistribution(),
                new Tuner\MagentoPhpCodeNodeSetter(new Tuner\Filter\MagentoAssetsAndExtendedLibs($path)),
                new Tuner\PhpCodeDeclaresAttributeSetter($phpElementsRegistry),
                new Tuner\PackagePathAttributeSetter(),
                new Tuner\DiXmlNodeSetter(),
                new Tuner\ProjectRootDiXmlNodeSetter(),
                new Tuner\DbSchemaNodeSetter(),
                new Tuner\ProjectRootDbSchemaNodeSetter()
            ],
            [
                new MagentoComponentsDiscovery($componentsIndex),
                new PhpCodeRuntimeTypesEstimation($phpElementsRegistry),
                new PhpCodeDiscovery($phpCodeIndex),
                new MagentoApiDiscovery($apiIndex),
                new MagentoVirtualTypesDiscovery($virtualTypes),
                new MagentoDbSchemaDiscovery($dbSchema)
            ]
        );

        $idx->index($path);

        return new MagentoReleaseIndexResult(
            $version,
            $componentsIndex,
            $apiIndex,
            $phpCodeIndex,
            $dictionary,
            $virtualTypes,
            $dbSchema
        );
    }
}
