<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\FullQualifiedStructuralElementName;

abstract class ClassLikeMember implements PhpStructuralElement
{
    /** @var object  */
    protected $cl;
    /** @var array  */
    protected $attributes = [];

    /**
     * @param ClassLikeDeclaration $cl
     */
    public function __construct(
        ClassLikeDeclaration $cl
    ) {
        $this->cl = (object)[
            'namespacedName' => $cl->getName(),
        ];
    }

    /**
     * @return FullQualifiedStructuralElementName|null
     */
    public function getFqsen(): ?FullQualifiedStructuralElementName
    {
        $className = $this->getContainerFqsen();
        if (!isset($className)) {
            return null;
        }

        return $this->createAlias($className);
    }

    /**
     * @param PhpFqsen\ClassLike $className
     * @return FullQualifiedStructuralElementName
     */
    abstract public function createAlias(PhpFqsen\ClassLike $className): FullQualifiedStructuralElementName;

    /**
     * @return PhpFqsen\ClassLike|null
     */
    public function getContainerFqsen(): ?PhpFqsen\ClassLike
    {
        if (isset($this->cl->namespacedName)) {
            return new PhpFqsen\ClassLike((string)$this->cl->namespacedName);
        }
        return null;
    }

    /**
     * @param string $attr
     * @param mixed $val
     */
    public function setAttribute(string $attr, $val): void
    {
        $this->attributes[$attr] = $val;
    }

    /**
     * @param string $attr
     * @param mixed $default
     * @return mixed|null
     */
    public function getAttribute(string $attr, $default = null)
    {
        return $this->attributes[$attr] ?? $default;
    }

    /**
     * @return array
     */
    public function getAttributes(): array
    {
        return $this->attributes;
    }

    /**
     * @return bool
     */
    abstract public function isPrivate(): bool;
}
