<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\MethodDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ParameterDeclaration;
use Magento\Mray\CodeStructuralElement\FullQualifiedStructuralElementName;
use phpDocumentor\Reflection\Type;
use function array_map;

class ClassLikeMethod extends ClassLikeMember
{
    /**
     * @var object
     */
    private $m;

    /**
     * @param ClassLikeDeclaration $cl
     * @param MethodDeclaration $m
     */
    public function __construct(
        ClassLikeDeclaration $cl,
        MethodDeclaration $m
    ) {
        parent::__construct($cl);
        $this->m = (object)[
            'name' => $m->getName(),
            'checksum' => $m->getImplementationChecksum(),
            'returnType' => $m->getReturnType(),
            'isPrivate' => $m->isPrivate(),
            'isPublic' => $m->isPublic(),
            'isAbstract' => $m->isAbstract(),
            'isFinal' => $m->isFinal(),
            'isStatic' => $m->isStatic(),
            'params' => array_map(function (ParameterDeclaration $pDecl) {
                return (object)[
                    'name' => $pDecl->getName(),
                    'variadic' => $pDecl->isVariadic(),
                    'optional' => $pDecl->isOptional(),
                    'type' => $pDecl->getExpectedArgumentType(),
                ];
            }, $m->getParams())
        ];
    }

    /**
     * @param PhpFqsen\ClassLike $className
     * @return FullQualifiedStructuralElementName
     */
    public function createAlias(PhpFqsen\ClassLike $className): FullQualifiedStructuralElementName
    {
        return new PhpFqsen\ClassLikeMethod(
            $className->pretty(),
            $this->m->name
        );
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return 'Php_Method';
    }

    /**
     * @return Type
     */
    public function getReturnType(): Type
    {
        return $this->m->returnType;
    }

    /**
     * @return bool
     */
    public function isPrivate(): bool
    {
        return $this->m->isPrivate;
    }

    /**
     * @return bool
     */
    public function isPublic(): bool
    {
        return $this->m->isPublic;
    }

    /**
     * @return bool
     */
    public function isAbstract(): bool
    {
        return $this->m->isAbstract;
    }

    /**
     * @return bool
     */
    public function isFinal(): bool
    {
        return $this->m->isFinal;
    }

    /**
     * @return bool
     */
    public function isStatic(): bool
    {
        return $this->m->isStatic;
    }

    /**
     * @return array
     */
    public function getParameters(): array
    {
        return $this->m->params;
    }

    /**
     * @return string|null
     */
    public function getChecksum(): ?string
    {
        return $this->m->checksum;
    }
}
