<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver;

use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;
use PhpParser\Node\Expr;
use function array_unique;
use function array_values;

class ArithmeticOperators implements Resolver
{
    /**
     * @var Memorized
     */
    private $memorized;

    /**
     * ArithmeticOperators constructor.
     */
    public function __construct()
    {
        $this->memorized = new Memorized();
    }

    /**
     * @param Expr $expr
     * @return Type
     */
    public function resolve(Expr $expr): Type
    {
        $operandTypes = [];
        foreach ($expr->getSubNodeNames() as $operand) {
            $operandTypes[] = $this->memorized->resolve($expr->{$operand});
        }
        $operandTypes = array_values(array_unique($operandTypes));

        $hasFloat = false;
        $hasInt = false;
        $hasArray = false;
        $hasOther = false;
        foreach ($operandTypes as $operandType) {
            if ($operandType instanceof Types\Float_) {
                $hasFloat = true;
            } elseif ($operandType instanceof Types\Integer) {
                $hasInt = true;
            } elseif ($operandType instanceof Types\Array_) {
                $hasArray = true;
            } else {
                $hasOther = true;
            }
        }

        // todo: move special cases to separate classes to avoid complex condition logic and provide possibility
        // todo: evolve implementation
        if ($hasArray) {
            if (($expr instanceof Expr\BinaryOp\Plus ||
                 $expr instanceof Expr\AssignOp\Plus) &&
                 !$hasInt &&
                 !$hasFloat
            ) {
                if ($hasOther) {
                    return TypeFactory::get(Types\Array_::class);
                }
                return $this->combineArrayTypes($operandTypes);
            }
            return TypeFactory::get(Types\Void_::class);
        }

        if ($expr instanceof Expr\BinaryOp\Mod || $expr instanceof Expr\AssignOp\Mod) {
            return TypeFactory::get(Types\Integer::class);
        }

        if ($hasFloat) {
            return TypeFactory::get(Types\Float_::class);
        }
        if ($expr instanceof Expr\BinaryOp\Div || $expr instanceof Expr\AssignOp\Div) {
            return TypeFactory::get(Types\Compound::class, [
                TypeFactory::get(Types\Integer::class),
                TypeFactory::get(Types\Float_::class),
            ]);
        }

        if ($hasInt && !$hasOther) {
            return TypeFactory::get(Types\Integer::class);
        }

        return TypeFactory::get(Types\Compound::class, [
            TypeFactory::get(Types\Integer::class),
            TypeFactory::get(Types\Float_::class),
        ]);
    }

    /**
     * @param Types\Array_[] $types
     * @return Types\Array_
     */
    private function combineArrayTypes(array $types): Types\Array_
    {
        $types = array_values($types);
        if (count($types) === 1) {
            return $types[0];
        }

        /** @var Types\Array_ $type */
        $type = TypeFactory::get(Types\Array_::class);
        return $type;
    }
}
