<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver;

use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ValueAccessor;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;
use PhpParser\Node\Expr;
use TypeError;
use function array_pop;
use function get_class;
use function sprintf;

class ArrayDimFetch implements Resolver, Contextual
{
    use Context;

    /**
     * @var Memorized
     */
    private $memorized;

    /**
     * ArrayDimFetch constructor.
     */
    public function __construct()
    {
        $this->memorized = new Memorized();
    }

    /**
     * @param Expr $expr
     * @return Type
     */
    public function resolve(Expr $expr): Type
    {
        if (!$expr instanceof Expr\ArrayDimFetch) {
            throw new TypeError(sprintf('%s is not applicable for %s expression.', self::class, get_class($expr)));
        }

        $type = $this->extractDimTypeFromArray($this->memorized->resolve($expr->var));
        if ($type) {
            return $type;
        }

        $typesStack = $this->readTypesStackFromContext(new ValueAccessor\Expr($expr));
        $dimType = array_pop($typesStack);
        if ($dimType && !$dimType instanceof Types\Mixed_) {
            return $dimType;
        }

        $arrayType = array_pop($typesStack);
        $dimType = null;
        if ($arrayType) {
            $dimType = $this->extractDimTypeFromArray($arrayType);
        }
        return $dimType ?? TypeFactory::get(Types\Mixed_::class);
    }

    /**
     * @param Type $arrayType
     * @return Type|null
     */
    private function extractDimTypeFromArray(Type $arrayType): ?Type
    {
        if ($arrayType instanceof Types\Array_ && !$arrayType->getValueType() instanceof Types\Mixed_) {
            return $arrayType->getValueType();
        }
        return null;
    }

    /**
     * @param ValueAccessor $accessor
     * @return Type[]
     */
    private function readTypesStackFromContext(ValueAccessor $accessor): array
    {
        $types = [];
        $container = $this->getContext();
        foreach ($accessor->getPath() as $pointer) {
            if ($pointer->getName() === '') {
                return [];
            }

            $var = $container->getVariable($pointer->getName());
            $types[] = $var->getType();
            $container = $var->getInnerContainer();
        }
        return $types;
    }
}
