<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver;

use Magento\Mray\CodeStructuralElement\Php\Reflection\MethodDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Type\ClassLikeMemberValueType as DeferredDecisionType;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ValueAccessor;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;
use PhpParser\Node\Expr;
use TypeError;
use function get_class;
use function sprintf;

class MethodReturnType implements Resolver, Contextual
{
    use StaticAccess;

    /**
     * @param Expr $expr
     * @return Type
     */
    public function resolve(Expr $expr): Type
    {
        if ($expr instanceof Expr\MethodCall || $expr instanceof Expr\NullsafeMethodCall) {
            $ct = $this->readTypeFromContext(
                new ValueAccessor\Expr($expr->var)
            ) ?: TypeFactory::get(Types\Mixed_::class);
            $mn = $expr->name instanceof Expr ? null : (string)$expr->name;
        } elseif ($expr instanceof Expr\StaticCall) {
            $ct = $this->getAccessedClass($expr);
            $mn = $expr->name instanceof Expr ? null : (string)$expr->name;
        } else {
            throw new TypeError(sprintf('%s is not applicable for %s expression.', self::class, get_class($expr)));
        }

        if (!isset($ct) || !isset($mn)) {
            return TypeFactory::get(Types\Mixed_::class);
        }

        return new DeferredDecisionType(
            $this->getContext()->getGlobal()->getStructuralElementDeclarations(),
            $ct,
            MethodDeclaration::class,
            $mn
        );
    }

    /**
     * @param ValueAccessor $accessor
     * @return Type|null
     */
    private function readTypeFromContext(ValueAccessor $accessor): ?Type
    {
        $type = null;
        $container = $this->getContext();
        foreach ($accessor->getPath() as $pointer) {
            $var = $container->getVariable($pointer->getName());
            $type = $var->getType();
            $container = $var->getInnerContainer();
        }
        return $type;
    }
}
