<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver;

use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use phpDocumentor\Reflection\Fqsen;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;
use PhpParser\Node\Name;
use PhpParser\Node\Expr;
use PhpParser\Node\Stmt;
use TypeError;
use function array_filter;
use function array_map;
use function count;
use function get_class;
use function sprintf;

class New_ implements Resolver
{
    /** @var Memorized  */
    private $memorized;

    /**
     * New_ constructor.
     */
    public function __construct()
    {
        $this->memorized = new Memorized();
    }

    /**
     * @param Expr $expr
     * @return Type
     */
    public function resolve(Expr $expr): Type
    {
        if (!$expr instanceof Expr\New_) {
            throw new TypeError(sprintf(
                '%s is not applicable for %s expression.',
                self::class,
                get_class($expr)
            ));
        }

        $class = $expr->class;
        if ($class instanceof Stmt\Class_) {
            return $this->resolveAnonymousClassInstantiation($class);
        } elseif ($class instanceof Name\FullyQualified) {
            return TypeFactory::get(Types\Object_::class, new Fqsen($class->toCodeString()));
        }

        return TypeFactory::get(Types\Object_::class);
    }

    /**
     * @param Stmt\Class_ $class
     * @return Type
     */
    private function resolveAnonymousClassInstantiation(Stmt\Class_ $class): Type
    {
        $baseTypes = [];
        if ($class->extends && $class->extends instanceof Name\FullyQualified) {
            $baseTypes[] = $class->extends->toCodeString();
        }

        $implements = array_filter($class->implements, function (Name $name) {
            return $name instanceof Name\FullyQualified;
        });
        foreach ($implements as $interface) {
            $baseTypes[] = $interface->toCodeString();
        }

        switch (count($baseTypes)) {
            case 0:
                return TypeFactory::get(Types\Object_::class);
            case 1:
                return TypeFactory::get(Types\Object_::class, new Fqsen($baseTypes[0]));
            default:
                return TypeFactory::get(Types\Compound::class, array_map(
                    function (string $type) {
                        return TypeFactory::get(Types\Object_::class, new Fqsen($type));
                    },
                    $baseTypes
                ));
        }
    }
}
