<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver;

use Magento\Mray\CodeStructuralElement\Php\Reflection\Context as ScopeContext;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use Magento\Mray\CodeStructuralElement\Php\Reflection\VariableDefinition\ThisDefinition;
use phpDocumentor\Reflection\Fqsen;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;
use PhpParser\ConstExprEvaluationException;
use PhpParser\ConstExprEvaluator;
use PhpParser\Node\Expr;
use PhpParser\Node\Name;
use TypeError;
use function is_string;
use function sprintf;

trait StaticAccess
{
    use Context;

    /** @var ConstExprEvaluator */
    private $evaluator;

    /**
     * StaticAccess constructor.
     */
    public function __construct()
    {
        $this->evaluator = new ConstExprEvaluator();
    }

    /**
     * @param Expr $expr
     * @return Type|null
     */
    private function getAccessedClass(Expr $expr): ?Type
    {
        if (!isset($expr->class)) {
            throw new TypeError(sprintf('%s is not an access to static member.', self::class));
        }

        if ($expr->class instanceof Expr) {
            try {
                $ct = $this->evaluator->evaluateSilently($expr->class);
                if (!is_string($ct)) {
                    return null;
                }
            } catch (ConstExprEvaluationException $e) {
                return null;
            }
            return TypeFactory::get(Types\Object_::class, new Fqsen('\\' . ltrim($ct, '\\')));
        }

        if ($expr->class instanceof Name && $expr->class->isSpecialClassName()) {
            $ctx = $this->getContext();
            if ($ctx instanceof ScopeContext\Pointer) {
                $ctx = $ctx->value();
            }
            if ($ctx instanceof ScopeContext\ClassLike || $ctx instanceof ScopeContext\ClassMethod) {
                $thisVar = $ctx->getThisVariableDefinition();
            } else {
                $thisVar = $ctx->getVariable('this');
            }

            if (!$thisVar instanceof ThisDefinition) {
                return null;
            }

            if (!$thisVar) {
                return null;
            }

            switch ((string)$expr->class) {
                case 'static':
                case 'self':
                    return $thisVar->getType();
                case 'parent':
                    return $thisVar->getParentType();
                default:
                    return null;
            }
        }
        return TypeFactory::get(Types\Object_::class, new Fqsen('\\' . ltrim((string) $expr->class, '\\')));
    }
}
