<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\NodeVisitor;

use Magento\Mray\CodeStructuralElement\Php\Reflection\CallableDeclaration\Function_;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration\Class_;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration\Interface_;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ConstantDeclaration\Const_;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Context\StructuralElementDeclarations;
use Magento\Mray\CodeStructuralElement\Php\Reflection\StructuralElementDeclaration;
use PhpParser\Node;
use PhpParser\NodeTraverser;
use PhpParser\NodeVisitor;

class DeclaresAttributeSetter implements NodeVisitor
{
    /** @var array  */
    public $fileScopeDeclarations = [];
    
    /** @var array  */
    public $allDeclarations = [];
    
    /** @var StructuralElementDeclarations|null  */
    private $elementsRegistry;

    /**
     * @param StructuralElementDeclarations|null $elementsRegistry
     */
    public function __construct(?StructuralElementDeclarations $elementsRegistry)
    {
        $this->elementsRegistry = $elementsRegistry;
    }

    /**
     * @param array $nodes
     * @return null
     */
    public function beforeTraverse(array $nodes)
    {
        $this->fileScopeDeclarations = [];
        return null;
    }

    /**
     * @param Node $node
     * @return int|null
     */
    public function enterNode(Node $node): ?int
    {
        if ($node instanceof Node\Stmt\Interface_) {
            $this->registerDeclaration(new Interface_($node, $this->elementsRegistry));
            return NodeTraverser::DONT_TRAVERSE_CURRENT_AND_CHILDREN;
        } elseif ($node instanceof Node\Stmt\Class_) {
            $this->registerDeclaration(new Class_($node, $this->elementsRegistry));
            return NodeTraverser::DONT_TRAVERSE_CURRENT_AND_CHILDREN;
        } elseif ($node instanceof Node\Stmt\Function_) {
            $this->registerDeclaration(new Function_($node));
            return NodeTraverser::DONT_TRAVERSE_CURRENT_AND_CHILDREN;
        } elseif ($node instanceof Node\Stmt\Const_) {
            foreach ($node->consts as $const) {
                $this->registerDeclaration(new Const_($node, $const));
            }
            return NodeTraverser::DONT_TRAVERSE_CURRENT_AND_CHILDREN;
        } elseif (!$node instanceof Node\Stmt) {
            return NodeTraverser::DONT_TRAVERSE_CURRENT_AND_CHILDREN;
        }
        return null;
    }

    /**
     * @param StructuralElementDeclaration $declaration
     */
    private function registerDeclaration(StructuralElementDeclaration $declaration): void
    {
        $declarationName = $declaration->getName();
        if (!$declarationName) {
            return;
        }

        // we cannot prevent multiple declarations of same element
        $this->fileScopeDeclarations[$declarationName][] = $declaration;
        $this->allDeclarations[$declarationName][] = $declaration;
        if ($this->elementsRegistry && $declaration instanceof ClassLikeDeclaration) {
            $this->elementsRegistry->registerClassLike($declaration);
        }
    }
    
    /**
     * @inheritDoc
     */
    public function leaveNode(Node $node)
    {
        return null;
    }

    /**
     * @inheritDoc
     */
    public function afterTraverse(array $nodes): ?array
    {
        return null;
    }
}
