<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\NodeVisitor;

use Closure;
use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Contextual;
use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Memorizing;
use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Resolver;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Context;
use PhpParser\Node;
use PhpParser\NodeVisitor;

class ExpressionRuntimeTypeResolver implements NodeVisitor
{
    /**
     * @var Memorizing|Resolver
     */
    private $resolver;
    /**
     * @var Closure
     */
    private $initContext;
    /**
     * @var Closure
     */
    private $restoreContext;

    /**
     * @param Resolver $resolver
     * @param Context|null $context
     */
    public function __construct(Resolver $resolver, ?Context $context = null)
    {
        $this->resolver = new Memorizing($resolver);
        if ($context && $resolver instanceof Contextual) {
            $originalContext = $resolver->getContext();
            $this->initContext = function () use ($resolver, $context) {
                $resolver->setContext($context);
            };
            $this->restoreContext = function () use ($resolver, $originalContext) {
                $resolver->setContext($originalContext);
            };
        } else {
            $this->initContext = $this->restoreContext = function () {
            };
        }
    }

    /**
     * @inheritDoc
     */
    public function beforeTraverse(array $nodes)
    {
        ($this->initContext)();
        return null;
    }

    /**
     * @inheritDoc
     */
    public function enterNode(Node $node)
    {
        return null;
    }

    /**
     * @inheritDoc
     */
    public function leaveNode(Node $node)
    {
        if ($node instanceof Node\Expr) {
            $this->resolver->resolve($node);
        }
        return null;
    }

    /**
     * @inheritDoc
     */
    public function afterTraverse(array $nodes)
    {
        ($this->restoreContext)();
        return null;
    }
}
