<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\NodeVisitor;

use Magento\Mray\CodeStructuralElement\Php\Parser\PhpDocParser;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use phpDocumentor\Reflection\Types\Context;
use PhpParser\Node;
use PhpParser\NodeVisitor;
use PHPStan\PhpDocParser\Ast\Node as PhpDocNode;
use PHPStan\PhpDocParser\Ast\Type\IdentifierTypeNode;
use PHPStan\PhpDocParser\Lexer\Lexer;
use PHPStan\PhpDocParser\Parser\ConstExprParser;
use PHPStan\PhpDocParser\Parser\TokenIterator;
use PHPStan\PhpDocParser\Parser\TypeParser;
use function get_object_vars;
use function is_array;

class PhpDocAst implements NodeVisitor
{
    public const PHP_DOC_COMMENT = 'docComment';

    /** @var PhpDocParser  */
    private $parser;
    /** @var Lexer  */
    private $lexer;

    /** @var string|null  */
    private $ns = null;
    /** @var array  */
    private $nsAliases = [];

    /**
     * PhpDocAst constructor.
     */
    public function __construct()
    {
        $this->lexer = new Lexer();
        $constExprParser = new ConstExprParser();
        $this->parser = new PhpDocParser(new TypeParser($constExprParser), $constExprParser);
    }

    /**
     * @inheritDoc
     */
    public function beforeTraverse(array $nodes)
    {
        return null;
    }

    /**
     * @inheritDoc
     */
    public function enterNode(Node $node)
    {
        if ($node instanceof Node\Stmt\Namespace_) {
            $this->ns = (string)$node->name;
            $this->nsAliases = [];
        } elseif ($node instanceof Node\Stmt\Use_) {
            foreach ($node->uses as $use) {
                $this->nsAliases[(string)$use->getAlias()] = (string)$use->name;
            }
        } elseif ($node instanceof Node\Stmt\GroupUse) {
            foreach ($node->uses as $use) {
                $this->nsAliases[(string)$use->getAlias()] = (string)Node\Name::concat($node->prefix, $use->name);
            }
        }

        if ($node instanceof Node\Expr\Assign && $node->hasAttribute('comments')) {
            if ((
                    $node->expr instanceof Node\Expr\Closure ||
                    $node->expr instanceof Node\Expr\ArrowFunction
                ) &&
                !$node->expr->hasAttribute('comments')
            ) {
                $node->expr->setAttribute('comments', $node->getAttribute('comments'));
            }
            return null;
        }

        if (!(
            $node instanceof Node\Stmt ||
            $node instanceof Node\Expr\Closure ||
            $node instanceof Node\Expr\ArrowFunction
        )) {
            return null;
        }

        $docComment = $node->getDocComment();
        if (!$docComment) {
            return null;
        }
        $parsed = $this->parser->parse(
            new TokenIterator($this->lexer->tokenize($docComment->getText()))
        );

        $context = $this->ns ? new Context($this->ns, $this->nsAliases) : null;
        $this->normalizeTypes($parsed->children, $context);

        $node->setAttribute(self::PHP_DOC_COMMENT, $parsed);
        return null;
    }

    /**
     * @param array $parsed
     * @param Context|null $context
     */
    private function normalizeTypes(array $parsed, ?Context $context): void
    {
        foreach ($parsed as $n) {
            if ($n instanceof IdentifierTypeNode) {
                $n->name = (string)TypeFactory::cast($n->name, $context);
                continue;
            } elseif ($n instanceof PhpDocNode) {
                $this->normalizeTypes(get_object_vars($n), $context);
            } elseif (is_array($n)) {
                $this->normalizeTypes($n, $context);
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function leaveNode(Node $node)
    {
        if ($node instanceof Node\Stmt\Namespace_) {
            $this->ns = null;
            $this->nsAliases = [];
        }
        return null;
    }

    /**
     * @inheritDoc
     */
    public function afterTraverse(array $nodes)
    {
        return null;
    }
}
