<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\NodeVisitor;

use Magento\Mray\CodeStructuralElement\Php\ClassLikeMember;
use Magento\Mray\CodeStructuralElement\Php\PhpStructuralElement;
use Magento\Mray\CodeStructuralElement\Php\PhpStructuralElementRegistry;
use Magento\Mray\CodeStructuralElement\Php\PhpStructuralElementRelationType;
use Magento\Mray\CodeStructuralElement\Php\ClassLike;
use Magento\Mray\CodeStructuralElement\Php\Class_;
use Magento\Mray\CodeStructuralElement\Php\Interface_;
use Magento\Mray\CodeStructuralElement\Php\ClassLikeMethod;
use Magento\Mray\CodeStructuralElement\Php\ClassLikeProperty;
use Magento\Mray\CodeStructuralElement\Php\ClassLikeConstant;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ConstantDeclaration\ClassConst;
use Magento\Mray\CodeStructuralElement\Php\Reflection\CallableDeclaration\Method;
use Magento\Mray\CodeStructuralElement\Php\Reflection\CallableDeclaration\PhpDocMethodTag;
use Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration\PhpDocPropertyTag;
use Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration\Property;
use Magento\Mray\CodeStructuralElement\FullQualifiedStructuralElementName;
use Magento\Mray\CodeStructuralElement\StructuralElementRelation;
use PhpParser\Node;
use PhpParser\NodeTraverser;
use PhpParser\NodeVisitor;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocNode;
use function array_filter;
use function array_merge;
use function count;
use function explode;
use function get_object_vars;
use function ucfirst;

// todo: rewrite
class StructuralElementRegistrar implements NodeVisitor
{
    /** @var PhpStructuralElementRegistry  */
    private $registry;

    /** @var bool */
    private $capturePrivateElements;
    /** @var bool */
    private $captureMethodImplementation;
    /** @var array  */
    private $location;

    /**
     * @param PhpStructuralElementRegistry $registry
     * @param bool $capturePrivateElements
     * @param bool $captureMethodImplementation
     */
    public function __construct(
        PhpStructuralElementRegistry $registry,
        bool $capturePrivateElements = false,
        bool $captureMethodImplementation = false
    ) {
        $this->registry = $registry;
        $this->capturePrivateElements = $capturePrivateElements;
        $this->captureMethodImplementation = $captureMethodImplementation;
    }

    /**
     * @param array $location
     */
    public function setLocation(array $location)
    {
        $this->location = $location;
    }

    /**
     * @inheritDoc
     */
    public function beforeTraverse(array $nodes)
    {
        return null;
    }

    /**
     * @inheritDoc
     */
    public function enterNode(Node $node)
    {
        if (!$node instanceof Node\Stmt) {
            return NodeTraverser::DONT_TRAVERSE_CURRENT_AND_CHILDREN;
        } elseif ($node instanceof Node\Stmt\ClassLike) {
            return NodeTraverser::DONT_TRAVERSE_CHILDREN;
        }
        return null;
    }

    /**
     * @inheritDoc
     */
    public function leaveNode(Node $node)
    {

        if (!$node instanceof Node\Stmt\ClassLike || $node instanceof Node\Stmt\Trait_) {
            // todo: implement traits support
            return null;
        }
        if (!isset($node->name, $node->namespacedName)) {
            // todo: implement anonymous classes support
            return null;
        }

        if ($node instanceof Node\Stmt\Class_) {
            $clDecl = new ClassLikeDeclaration\Class_($node);
            $cl = $this->initElement(new Class_($clDecl), [$node]);
            $inherits = [];
            $classFqsen = $cl->getFqsen();
            $extends = $cl->getParentClassFqsen();
            if ($extends) {
                $inherits[] = $extends;
                $this->registry->registerRelation(new StructuralElementRelation(
                    $classFqsen,
                    $extends,
                    PhpStructuralElementRelationType\ExtendsClass::type()
                ));
            }
            foreach ($cl->getImplementedInterfacesFqsen() as $implements) {
                $inherits[] = $implements;
                $this->registry->registerRelation(new StructuralElementRelation(
                    $classFqsen,
                    $implements,
                    PhpStructuralElementRelationType\Implements_::type()
                ));
            }

        } elseif ($node instanceof Node\Stmt\Interface_) {
            $clDecl = new ClassLikeDeclaration\Interface_($node);
            $cl = $this->initElement(new Interface_($clDecl), [$node]);
            $interfaceFqsen = $cl->getFqsen();
            $inherits = [];
            foreach ($cl->getExtendedInterfacesFqsen() as $extends) {
                $inherits[] = $extends;
                $this->registry->registerRelation(new StructuralElementRelation(
                    $interfaceFqsen,
                    $extends,
                    PhpStructuralElementRelationType\InterfaceExtends::type()
                ));
            }

        }

        if (isset($clDecl, $cl)) {
            /** @var ClassLikeMember $classLikeMember */
            foreach ($this->getMemberElements($clDecl, $node) as $classLikeMember) {
                if ($this->capturePrivateElements || !$classLikeMember->isPrivate()) {
                    $this->registry->register($classLikeMember);
                    $this->registry->registerRelation(new StructuralElementRelation(
                        $classLikeMember->getContainerFqsen(),
                        $classLikeMember->getFqsen(),
                        PhpStructuralElementRelationType\DeclaresMember::type()
                    ));
                }
            }

            $this->applyNewKnowledgeAbout($cl, $inherits ?? []);
            $this->registry->register($cl);
        }

        return null;
    }

    /**
     * @inheritDoc
     */
    public function afterTraverse(array $nodes)
    {
        return null;
    }

    /**
     * @param ClassLikeDeclaration $clDef
     * @param Node\Stmt\ClassLike $node
     * @return iterable
     */
    private function getMemberElements(ClassLikeDeclaration $clDef, Node\Stmt\ClassLike $node): iterable
    {
        /** @var PhpDocNode $phpDoc */
        $phpDoc = $node->getAttribute(PhpDocAst::PHP_DOC_COMMENT);
        if ($phpDoc) {
            foreach ($phpDoc->getPropertyTagValues() as $prop) {
                yield new ClassLikeProperty(
                    $clDef,
                    new PhpDocPropertyTag($prop, $clDef->getName())
                );
            }
            foreach ($phpDoc->getPropertyReadTagValues() as $prop) {
                yield new ClassLikeProperty(
                    $clDef,
                    new PhpDocPropertyTag($prop, $clDef->getName())
                );
            }
            foreach ($phpDoc->getPropertyWriteTagValues() as $prop) {
                yield new ClassLikeProperty(
                    $clDef,
                    new PhpDocPropertyTag($prop, $clDef->getName())
                );
            }
            foreach ($phpDoc->getMethodTagValues() as $method) {
                yield new ClassLikeMethod(
                    $clDef,
                    new PhpDocMethodTag($method, $clDef->getName())
                );
            }
        }
        foreach ($node->getConstants() as $constsGroupNode) {
            foreach ($constsGroupNode->consts as $constNode) {
                yield $this->initElement(new ClassLikeConstant(
                    $clDef,
                    new ClassConst($constsGroupNode, $constNode, $clDef->getName())
                ), [$node, $constsGroupNode, $constNode]);
            }
        }
        foreach ($node->getProperties() as $propsGroupNode) {
            foreach ($propsGroupNode->props as $propNode) {
                 yield $this->initElement(new ClassLikeProperty(
                     $clDef,
                     new Property($propsGroupNode, $propNode, $clDef->getName())
                 ), [$node, $propsGroupNode, $propNode]);
            }
        }
        foreach ($node->getMethods() as $methodNode) {
             yield $this->initElement(new ClassLikeMethod(
                 $clDef,
                 new Method($methodNode, $clDef->getName())
             ), [$node, $methodNode]);
        }
        return;
        // phpcs:ignore Squiz.PHP.NonExecutableCode
        yield;
    }

    /**
     * @param PhpStructuralElement $element
     * @param array $sourceNodes
     * @return PhpStructuralElement
     */
    private function initElement(PhpStructuralElement $element, array $sourceNodes): PhpStructuralElement
    {
        $defNode = $sourceNodes[count($sourceNodes) - 1] ?? null;
        if (!empty($this->location)) {
            $element->setAttribute('location', array_merge(
                $this->location,
                $defNode ?
                    [
                        'position' => [
                            'startLine' => $defNode->getAttribute('startLine'),
                            'endLine' => $defNode->getAttribute('endLine'),
                            'startFilePos' => $defNode->getAttribute('startFilePos'),
                            'endFilePos' => $defNode->getAttribute('endFilePos'),
                        ]
                    ] :
                    []
            ));
        }
        if ($this->captureMethodImplementation && $defNode instanceof Node\Stmt\ClassMethod && $defNode->stmts) {
            $element->setAttribute('implementation', $defNode->stmts);
        }

        // todo: delegate logic to injectable handlers
        foreach ($sourceNodes as $node) {
            // todo: move Magento-specific code to separate injectable handler
            if (isset($node->magento)) {
                foreach (get_object_vars($node->magento) as $section => $val) {
                    $element->{'magento' . ucfirst($section)} = $val;
                }
            }
        }
        return $element;
    }

    /**
     * @param ClassLike $classLike
     * @param array $parents
     */
    private function applyNewKnowledgeAbout(ClassLike $classLike, array $parents)
    {
        $members = $this->pullMembersFromParents($classLike, $parents);
        $this->pushMembersToChildren($classLike, $members);
    }

    /**
     * @param ClassLike $classLike
     * @param array $parents
     * @return array
     */
    private function pullMembersFromParents(ClassLike $classLike, array $parents): array
    {
        $donors = [];
        $ownMembers = [];
        $allMembers = [];

        foreach ($parents as $parent) {
            $this->updateInheritanceRelations($classLike->getFqsen(), $parent);
            $donors[(string)$parent] = $this->registry->find($parent);
        }
        $donors = array_filter($donors);

        foreach ($this->registry->findRelationsInitiatedBy($classLike->getFqsen()) as $ser) {
            if ($ser->isCaseOf(PhpStructuralElementRelationType\DeclaresMember::class)) {
                $allMembers[(string)$ser->getInvolved()] =
                $ownMembers[$this->extractMemberName($ser->getInvolved())] =
                    $this->registry->find($ser->getInvolved());
            }
            if ($ser->isCaseOf(PhpStructuralElementRelationType\InheritsMember::class)) {
                $allMembers[(string)$ser->getInvolved()] = $this->registry->find($ser->getInvolved());
            }
        }

        foreach ($donors as $donor) {
            foreach ($this->findMembersOf($donor->getFqsen()) as $donorMember) {
                // todo: ensure multiple interfaces inheritance works
                $memberName = $this->extractMemberName($donorMember->getFqsen());
                if (isset($ownMembers[$memberName])) {
                    $this->reassignData($donorMember, $ownMembers[$memberName]);
                }

                if (isset($allMembers[(string)$donorMember->getFqsen()])) {
                    continue; // previously known relation
                }

                $this->registry->registerRelation(new StructuralElementRelation(
                    $classLike->getFqsen(),
                    $donorMember->getFqsen(),
                    PhpStructuralElementRelationType\InheritsMember::type()
                ));
                $allMembers[(string)$donorMember->getFqsen()] = $donorMember;
                if (!isset($ownMembers[$memberName])) {
                    $this->registry->registerAlias($donorMember->createAlias($classLike->getFqsen()), $donorMember);
                }
            }
        }

        return $allMembers;
    }

    /**
     * @param ClassLike $classLike
     * @param array $members
     */
    private function pushMembersToChildren(ClassLike $classLike, array $members): void
    {
        if (empty($members)) {
            return;
        }

        foreach ($this->registry->findRelationsInvolve($classLike->getFqsen()) as $ser) {
            if ($ser->isCaseOf(PhpStructuralElementRelationType\ChildOf::class)) {
                $child = $this->registry->find($ser->getInitiator());
                $this->updateInheritanceRelations($child->getFqsen(), $classLike->getFqsen());
                if (!$child) { // will be handled when child definition processed
                    continue;
                }
                $childMembers = [];
                foreach ($this->findMembersOf($ser->getInitiator()) as $childMember) {
                    $childMembers[$this->extractMemberName($childMember->getFqsen())] = $childMember;
                }
                foreach ($members as $parentMember) {
                    $this->registry->registerRelation(new StructuralElementRelation(
                        $child->getFqsen(),
                        $parentMember->getFqsen(),
                        PhpStructuralElementRelationType\InheritsMember::type()
                    ));

                    if (!isset($childMembers[$this->extractMemberName($parentMember->getFqsen())])) {
                        $this->registry->registerAlias($parentMember->createAlias($child->getFqsen()), $parentMember);
                    } else {
                        $this->reassignData(
                            $parentMember,
                            $childMembers[$this->extractMemberName($parentMember->getFqsen())]
                        );
                    }
                }
                $this->pushMembersToChildren($child, $members); // push all to reassign data if needed
            }
        }
    }

    /**
     * @param object $donor
     * @param object $recipient
     */
    private function reassignData(object $donor, object $recipient)
    {
        foreach (get_object_vars($donor) as $prop => $val) {
            if (!isset($recipient->{$prop})) {
                $recipient->{$prop} = $val;
            }
        }
    }

    /**
     * @param FullQualifiedStructuralElementName $fqsen
     * @return iterable
     */
    private function findMembersOf(FullQualifiedStructuralElementName $fqsen): iterable
    {
        foreach ($this->registry->findRelationsInitiatedBy($fqsen) as $ser) {
            if (!$ser->isCaseOf(PhpStructuralElementRelationType\ContainsMember::class)) {
                continue;
            }
            $member = $this->registry->find($ser->getInvolved());
            if ($member) {
                yield $member;
            }
        }
        return;
        // phpcs:ignore Squiz.PHP.NonExecutableCode
        yield;
    }

    /**
     * @param FullQualifiedStructuralElementName $child
     * @param FullQualifiedStructuralElementName $parent
     */
    private function updateInheritanceRelations(
        FullQualifiedStructuralElementName $child,
        FullQualifiedStructuralElementName $parent
    ) {
        $this->registry->registerRelation(new StructuralElementRelation(
            $child,
            $parent,
            PhpStructuralElementRelationType\Inherits::type()
        ));
        foreach ($this->registry->findRelationsInitiatedBy($parent) as $parentRel) {
            if ($parentRel->isCaseOf(PhpStructuralElementRelationType\Inherits::class)) {
                $this->updateInheritanceRelations($child, $parentRel->getInvolved()); // grandparent
            }
        }
    }

    /**
     * @param FullQualifiedStructuralElementName $fqsen
     * @return mixed|string
     */
    private function extractMemberName(FullQualifiedStructuralElementName $fqsen)
    {
        $fqsen = (string)$fqsen;
        $fqsen = explode('::', $fqsen, 2);
        return $fqsen[count($fqsen) - 1];
    }
}
