<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\NodeVisitor;

use Magento\Mray\CodeStructuralElement\Php\Reflection\Context;
use Magento\Mray\CodeStructuralElement\Php\Reflection\VariableDefinition;
use PhpParser\Node;
use PhpParser\NodeVisitor;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocNode;

class VariableDeclarationReflector implements NodeVisitor
{
    /** @var Context  */
    private $context;

    /**
     * @param Context $context
     */
    public function __construct(Context $context)
    {
        $this->context = $context;
    }

    /**
     * @inheritDoc
     */
    public function beforeTraverse(array $nodes)
    {
        return null;
    }

    /**
     * @inheritDoc
     */
    public function enterNode(Node $node)
    {
        if ($node->hasAttribute(PhpDocAst::PHP_DOC_COMMENT)) {
            if ($node instanceof Node\Stmt\Expression || $node instanceof Node\Stmt\Nop) {
                $this->readVariableDefinitionsFromPhpDoc($node->getAttribute(PhpDocAst::PHP_DOC_COMMENT));
            } elseif ($node instanceof Node\Stmt\Function_ ||
                $node instanceof Node\Stmt\ClassMethod ||
                $node instanceof Node\Expr\Closure ||
                $node instanceof Node\Expr\ArrowFunction
            ) {
                $this->readParameterDefinitionsFromPhpDoc($node->getAttribute(PhpDocAst::PHP_DOC_COMMENT));
            }
        }

        if ($node instanceof Node\Expr\ClosureUse) {
            $this->context->registerVariableDefinition(new VariableDefinition\ClosureUse($node));
        } elseif ($node instanceof Node\Stmt\Global_) {
            foreach ($node->vars as $var) {
                $this->context->registerVariableDefinition(new VariableDefinition\GlobalVariableUse($node, $var));
            }
        }

        return null;
    }

    /**
     * @inheritDoc
     */
    public function leaveNode(Node $node)
    {
        if ($node instanceof Node\Expr\Assign || $node instanceof Node\Expr\AssignRef) {
            $this->context->registerVariableDefinition(new VariableDefinition\Assignment($node));
            if ($node->expr instanceof  Node\Expr\Array_) {
                foreach ($node->expr->items as $arrayItem) {
                    $this->context->registerVariableDefinition(
                        new VariableDefinition\ArrayAssignment($node, $arrayItem)
                    );
                }
            }
        } elseif ($node instanceof Node\Param) {
            $this->context->registerVariableDefinition(new VariableDefinition\Param($node));
        }
        return null;
    }

    /**
     * @param PhpDocNode|null $phpDoc
     */
    private function readVariableDefinitionsFromPhpDoc(?PhpDocNode $phpDoc): void
    {
        if (!$phpDoc) {
            return;
        }

        foreach ($phpDoc->getVarTagValues() as $varTagValue) {
            $this->context->registerVariableDefinition(new VariableDefinition\PhpDocVarTag($varTagValue));
        }
        foreach ($phpDoc->getVarTagValues('@xglobal') as $varTagValue) {
            $this->context->getGlobal()->registerSuperGlobalVariableDefinition(
                new VariableDefinition\PhpDocVarTag($varTagValue)
            );
        }
    }

    /**
     * @param PhpDocNode|null $phpDoc
     */
    private function readParameterDefinitionsFromPhpDoc(?PhpDocNode $phpDoc): void
    {
        if (!$phpDoc) {
            return;
        }

        foreach ($phpDoc->getParamTagValues() as $paramTagValue) {
            $this->context->registerVariableDefinition(new VariableDefinition\PhpDocParamTag($paramTagValue));
        }
    }

    /**
     * @inheritDoc
     */
    public function afterTraverse(array $nodes)
    {
        return null;
    }
}
