<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\CallableDeclaration;

use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\PhpDocAst;
use Magento\Mray\CodeStructuralElement\Php\Reflection\CallableDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\InCodeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Type\ClassLikeMemberValueType;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\StatementNodeVisitor;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;
use PhpParser\Node;
use PhpParser\NodeDumper;
use PhpParser\NodeTraverser;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocNode;
use function array_filter;
use function array_unique;
use function array_values;
use function count;
use function sha1;
use function sprintf;

abstract class FunctionLike implements CallableDeclaration, InCodeDeclaration
{
    /**
     * @var Node\FunctionLike
     */
    private $node;

    /**
     * @param Node\FunctionLike $node
     */
    public function __construct(
        Node\FunctionLike $node
    ) {
        $this->node = $node;
    }

    /**
     * @inheritDoc
     */
    public function getParams(): array
    {
        $params = [];
        $annotatedTypes = function (?string $name): array {
            if (!$name) {
                return [];
            }
            $phpDoc = $this->getPhpDoc();
            if (!$phpDoc) {
                return [];
            }
            $name = '$' . $name;
            $types = [];
            foreach ($phpDoc->getParamTagValues() as $paramTagValue) {
                if ($paramTagValue->parameterName === $name) {
                    $types[] = TypeFactory::cast($paramTagValue->type);
                }
            }
            return $types;
        };

        foreach ($this->node->getParams() as $param) {
            $params[] = new Param($param, $annotatedTypes);
        }
        return $params;
    }

    /**
     * @return Type
     */
    public function getReturnType(): Type
    {
        $explicitlyDeclared = array_values(array_filter([
            $this->getReturnTypeFromHint(),
            $this->getReturnTypeFromDocBlock(),
        ]));
        switch (count($explicitlyDeclared)) {
            case 2:
                return TypeFactory::preciseType(...$explicitlyDeclared);
            case 1:
                return $explicitlyDeclared[0];
            default:
                return $this->getReturnTypeFromImplementation();
        }
    }

    /**
     * @return Type|null
     */
    private function getReturnTypeFromHint(): ?Type
    {
        $typeHint = $this->node->getReturnType();
        if ($typeHint) {
            return TypeFactory::cast($typeHint);
        }
        return null;
    }

    /**
     * @return Type|null
     */
    private function getReturnTypeFromDocBlock(): ?Type
    {
        $phpDoc = $this->getPhpDoc();
        if (!$phpDoc) {
            return null;
        }

        $type = null;
        foreach ($phpDoc->getReturnTagValues() as $returnTagValue) {
            $type = $returnTagValue->type;
        }

        if (isset($type)) {
            return TypeFactory::cast($type);
        }

        return null;
    }

    /**
     * @return PhpDocNode|null
     */
    public function getPhpDoc(): ?PhpDocNode
    {
        return $this->node->getAttribute(PhpDocAst::PHP_DOC_COMMENT);
    }

    /**
     * @return Type
     */
    protected function getReturnTypeFromImplementation(): Type
    {
        if (!isset($this->node->stmts)) {
            return TypeFactory::get(Types\Mixed_::class);
        }

        $traverser = new NodeTraverser();
        $visitor = new StatementNodeVisitor();

        $traverser->addVisitor($visitor);
        $traverser->traverse($this->node->stmts);

        $types = array_values(
            array_unique(
                array_map(
                    function (Type $return) {
                        return $this->isReturnRecursive($return) ? TypeFactory::get(Types\Mixed_::class) : $return;
                    },
                    $visitor->getReturns()
                )
            )
        );
        switch (count($types)) {
            case 0:
                return TypeFactory::get(Types\Void_::class);
            case 1:
                return $types[0];
            default:
                return TypeFactory::get(Types\Compound::class, $types);
        }
    }

    /**
     * Check if return is a result of the method, and that method is the same method as current node (recursion)
     *
     * @param Type $return
     * @return bool
     */
    private function isReturnRecursive(Type $return): bool
    {
        return $return instanceof ClassLikeMemberValueType &&
            ltrim((string) $return->getClassLikeType(), '\\') === ltrim($this->getClassLikeName(), '\\') &&
            $return->getElementName() === $this->getName();
    }

    /**
     * @return Type
     */
    public function getValueType(): Type
    {
        return $this->getReturnType();
    }

    /**
     * @return string|null
     */
    public function getImplementationChecksum(): ?string
    {
        return null;
        // phpcs:ignore Squiz.PHP.NonExecutableCode
        $dumper = new NodeDumper();
        // phpcs:ignore Squiz.PHP.NonExecutableCode
        return sprintf('v1:sha1:%s', sha1($dumper->dump($this->node)));
    }

    /**
     * @inheritDoc
     */
    public function getDeclarationNodes(): array
    {
        return [
            $this->node
        ];
    }
}
