<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\CallableDeclaration;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration\Finder;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeMemberPhpDocInheritance;
use Magento\Mray\CodeStructuralElement\Php\Reflection\MethodDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Type\CodeDiscoveryDependentType;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;
use PhpParser\Node;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocNode;
use function array_unique;
use function array_values;
use function count;

class Method extends FunctionLike implements MethodDeclaration
{
    use ClassLikeMemberPhpDocInheritance;

    /** @var Node\Stmt\ClassMethod  */
    private $node;
    /** @var string|null  */
    private $classLikeName;

    /**
     * @param Node\Stmt\ClassMethod $node
     * @param string|null $classLikeName
     * @param Finder|null $inheritedDeclarationsFinder
     */
    public function __construct(
        Node\Stmt\ClassMethod $node,
        ?string $classLikeName,
        ?Finder $inheritedDeclarationsFinder = null
    ) {
        $this->node = $node;
        $this->classLikeName = $classLikeName;
        $this->inheritedDeclarationsFinder = $inheritedDeclarationsFinder;

        parent::__construct($node);
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return (string)$this->node->name;
    }

    /**
     * @return string|null
     */
    public function getClassLikeName(): ?string
    {
        return $this->classLikeName;
    }

    /**
     * @return PhpDocNode|null
     */
    public function getPhpDoc(): ?PhpDocNode
    {
        return $this->getPhpDocWithResolvedInheritance(parent::getPhpDoc());
    }

    /**
     * @return Type|null
     */
    private function getInheritedReturnType(): ?Type
    {
        $rts = [];
        foreach ($this->findInheritedDeclarations() as $inheritedDeclaration) {
            $rt = $inheritedDeclaration->getReturnType();
            if (!$rt instanceof Types\Mixed_ && !$rt instanceof CodeDiscoveryDependentType) {
                $rts[] = $rt;
            }
        }
        if (empty($rts)) {
            return null;
        }
        $rts = array_values(array_unique($rts));
        if (count($rts) === 1) {
            return $rts[0];
        }
        return TypeFactory::get(Types\Compound::class, $rts);
    }

    /**
     * @return Type
     */
    protected function getReturnTypeFromImplementation(): Type
    {
        $ownImplementationType = parent::getReturnTypeFromImplementation();
        $inheritedReturnType = $this->getInheritedReturnType();
        if ($inheritedReturnType) {
            return TypeFactory::preciseType($inheritedReturnType, $ownImplementationType);
        }
        return $ownImplementationType;
    }

    /**
     * @return bool
     */
    public function isPublic(): bool
    {
        return $this->node->isPublic();
    }

    /**
     * @return bool
     */
    public function isProtected(): bool
    {
        return $this->node->isProtected();
    }

    /**
     * @return bool
     */
    public function isPrivate(): bool
    {
        return $this->node->isPrivate();
    }

    /**
     * @return bool
     */
    public function isAbstract(): bool
    {
        return $this->node->isAbstract();
    }

    /**
     * @return bool
     */
    public function isFinal(): bool
    {
        return $this->node->isFinal();
    }

    /**
     * @return bool
     */
    public function isStatic(): bool
    {
        return $this->node->isStatic();
    }

    /**
     * @return bool
     */
    public function isMagic(): bool
    {
        return $this->node->isMagic();
    }
}
