<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\CallableDeclaration;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ParameterDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use PhpParser\Node\Param as ParamNode;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;

class Param implements ParameterDeclaration
{
    /** @var ParamNode  */
    private $node;
    /** @var callable */
    private $typeSuggestionsProvider;

    /**
     * @param ParamNode $node
     * @param callable $typeSuggestionsProvider Accepts name of parameter and returns list of suggested types
     */
    public function __construct(ParamNode $node, callable $typeSuggestionsProvider)
    {
        $this->node = $node;
        $this->typeSuggestionsProvider = $typeSuggestionsProvider;
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return isset($this->node->var->name) ? (string)$this->node->var->name : null;
    }

    /**
     * @return Type
     */
    public function getType(): Type
    {
        $type = $this->getExpectedArgumentType();
        if ($this->isVariadic()) {
            $type = TypeFactory::get(Types\Array_::class, $type);
        }
        return $type;
    }

    /**
     * @return Type
     */
    public function getExpectedArgumentType(): Type
    {
        $type = TypeFactory::cast($this->node->type);
        foreach ($this->getSuggestedTypes() as $suggestedType) {
            $type = TypeFactory::preciseType($type, $suggestedType);
        }
        return $type;
    }

    /**
     * @return array
     */
    public function getSuggestedTypes(): array
    {
        $provider = $this->typeSuggestionsProvider;
        return $provider($this->getName());
    }

    /**
     * @return bool
     */
    public function isVariadic(): bool
    {
        return $this->node->variadic;
    }

    /**
     * @return bool
     */
    public function isOptional(): bool
    {
        return isset($this->node->default);
    }
}
